<?php

namespace App\DataTables\Agents;


use App\Models\Approval;
use App\Models\Currency;
use App\Models\LedgerApproval;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class ApprovalDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    protected $filters = [];
    protected $type = "in";
    public function  __construct($filters = [], $type = "in")
    {
        Log::info($filters);

        $this->filters = $filters;
        $this->type = $type;
    }
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('action', function ($data) {

                $type = $this->type;

                return view('AgentPanel.Transfer.Approval.actions', compact('data', 'type'));
            })
            ->addColumn("from", function ($query) {
                return $query->sender->name ?? 'N/A';
            })
            ->addColumn("to", function ($query) {
                return $query->receiver->name ?? 'N/A';
            })
            ->addColumn("debit_entry", function ($query) {
                $html = "";
                $amount = $query->amount;
                $currency = $query->SenderCurrency->name ?? 'N/A';
                $color = $query->SenderCurrency->color ?? 'N/A';
                $fee = $query->send_fee;
                $html .= "<span>" . PriceFormat($amount) . "</span><br>";
                $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
                $html .= "<span style='color:blue'>(" . PriceFormat($fee) . ")</span>";
                return $html;
            })

            ->addColumn("credit_entry", function ($query) {
                $html = "";
                $amount = $query->delivery_amount;
                $currency = $query->DeliveredCurrency->name ?? 'N/A';
                $color = $query->DeliveredCurrency->color ?? 'N/A';
                $fee = $query->receiver_fee ?? 0;
                $html .= "<span>" . PriceFormat($query->delivery_amount) . "</span><br>";
                $html .= "<span style='color:" . $color . "'>" . $currency . "</span><br>";
                $html .= "<span style='color:red'>(" . PriceFormat($fee) . ")</span>";
                return $html;
            })
            ->addColumn("amount", function ($query) {
                return PriceFormat($query->amount);
            })
            ->editColumn("status", function ($query) {
                return "<span class='badge bg-" . $query->status_color . "'>" . __($query->status) . "</span>";
            })
            ->addColumn("created_by", function ($query) {
                return $query->user->name ?? 'N/A';
            })
            ->addColumn("created_at", function ($query) {
                return $query->created_at->format('Y-m-d H:i');
            })
            ->editColumn("note", function ($query) {
                return $query->note ?? __("general.no_note");;
            })
            ->addColumn("profit", function ($query) {
                return PriceFormat($query->send_fee - $query->receive_fee);
            })
            ->addColumn("customer", function ($query) {
                return  $query->recipient_name ?? 'N/A';
            })
            ->rawColumns(['action', 'debit_entry', 'credit_entry', 'amount', 'status', 'created_at'])
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(LedgerApproval $model): QueryBuilder
    {
        Log::info("filters");
        Log::info($this->filters);
        $agent = auth()->guard('agent')->user()->agent->rootAgent();
        $query = $model->newQuery()
            ->with([

                'sender',
                'receiver',
                'DeliveredCurrency',
                'SenderCurrency',
                'user'
            ])
            ->orderBy('created_at', 'desc');
        $filters = $this->filters;
        if (request()->filled('filters')) {


            $filters = request()->get('filters');
        }
        Log::info($filters);
        $query = $query->Filter($filters);
        return $query;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->parameters([
                'language' => [
                    'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
                ],
                'processing' => true,
                'serverSide' => true,
                'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
            ])
            ->setTableId('approval-table')
            ->columns($this->getColumns())
            ->ajax([
                'url' => '', // current URL
                'type' => 'GET',
                'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }'
            ])
            ->dom('<"row"' .
                '<"col-md-2"<"ms-n2"l>>' .
                '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
                '>t' .
                '<"row"' .
                '<"col-sm-12 col-md-6"i>' .
                '<"col-sm-12 col-md-6"p>' .
                '>')


            ->buttons([

                Button::raw([
                    'text' => '<i class="ti ti-search"></i> ',
                    'className' => 'btn btn-info ms-2',
                    'attr' => [
                        'title' =>  __("general.search"),
                        'data-bs-toggle' => 'offcanvas',
                        'data-bs-target' => '#FilterModal',
                    ],
                ])
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [

            Column::make("created_at")
                ->title(__('transfers.created_at'))
                ->width('10%'),
            Column::make("debit_entry")
                ->title(__('transfers.debit'))
                ->width('10%')
                ->searchable(false),
            Column::make("from")
                ->title(__('transfers.from'))
                ->width('12%'),
            Column::make("customer")->title(__('transfers.receiver_client'))->width('12%'),
            Column::make("credit_entry")
                ->title(__('transfers.credit'))
                ->width('10%')
                ->searchable(false),
            Column::make("to")
                ->title(__('transfers.to'))
                ->width('12%'),
            Column::make("profit")
                ->title(__('transfers.profit'))
                ->width('5%'),
            Column::make("notes")
                ->title(__('transfers.notes'))
                ->width('10%'),
            Column::make("created_by")
                ->title(__('transfers.created_by'))
                ->width('10%'),
            Column::make("reference")
                ->title(__('transfers.reference'))
                ->width('20%'),

            Column::make("status")
                ->title(__('transfers.status'))
                ->width('20%'),







            Column::computed('action')
                ->title(__('Actions'))
                ->exportable(false)
                ->printable(false)
                ->width('6%')
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Approval_' . date('YmdHis');
    }
}
