<?php


namespace App\DataTables\Agents;

use App\Models\BalanceOverwrite;
use App\Models\Currency;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\Html\Column;

class BalanceDataTable extends DataTable
{
  protected array $currencies = [];
  protected bool $showRemittanceDetails = false;
  protected $filters = [];

  public function __construct()
  {
    $this->filters = session('agent_balance_selected_currencies')
      ? ['currencies' => session('agent_balance_selected_currencies')]
      :  Currency::get()->pluck('code')->toArray();
    $this->showRemittanceDetails = session('showBalancesDetails') ?? false;

    $this->currencies = BalanceOverwrite::whereIn('currency',  session('agent_balance_selected_currencies', Currency::get()->pluck('code')->toArray())) // or any logic
      ->pluck('currency')->unique()->sort()->values()->toArray();

    // ✅ Load filters early so getColumns() sees them

  }

  /*************  ✨ Windsurf Command ⭐  *************/
  /**
   * Get the query source of dataTable.
   *
   * This method constructs a query to retrieve balances from the BalanceOverwrite model.
   * It applies filters if available and logs the applied filters.
   * The query calculates the total credit and debit amounts for each ledger UUID and currency,
   * grouping the results by these fields and including related account names.
   * If currency filters are provided, the query is further restricted to those currencies.
   *
   * @return \Illuminate\Database\Eloquent\Builder The query builder instance.
   */
  public function query()
  {
    $agent = auth('agent')->user()->agent->rootAgent();
    $agent_childs = $agent->getallchildid();
    if (request()->filled('filters')) {


      $filters = request()->get('filters');

      Log::info('Filters applied:', $filters);
    }
    $query = BalanceOverwrite::selectRaw('

                ledgerUuid,
                currency,
                SUM(CASE WHEN balance > 0 THEN balance ELSE 0 END) as credit,
                SUM(CASE WHEN balance < 0 THEN ABS(balance) ELSE 0 END) as debit
            ')->wherehas("account", function ($query) use ($agent_childs) {
      $query->whereIn("extra->agent_id", $agent_childs)->where("extra->type", "cashier");
    })
      ->groupBy('ledgerUuid', 'currency');

    if (!empty($filters['currencies']) && is_array($filters['currencies'])) {
      $query->whereIn('currency', $filters['currencies']);
    }
    return $query;
  }

  public function dataTable($query)
  {
    $raw = $query->get();
    Log::info("showRemittanceDetails in query" . $this->showRemittanceDetails);
    $grouped = [];
    $totals = [];
    $remittanceTotals = [];
    $commissionTotals = [];

    foreach ($raw as $row) {

      $accountCode = $row->account->code ?? 'N/A';
      $accountName = $row->account->names->where("language", app()->getLocale())->first()->name ?? 'N/A';
      $currency = $row->currency;
      $credit = $row->credit;
      $debit = $row->debit;

      $key = $accountCode;
      $creditLimit = optional(optional(optional($row->account)->agent())->getTransferGlobalCreditLimit())->credit_limit ?? null;
      $agentBasedBalances = optional(optional($row->account)->agent)->TransferGlobalCreditLimit->credit_limit ?? null;

      // normalize agent balance to a numeric value (if array -> sum)
      $agentBalanceNumeric = null;
      if (is_array($agentBasedBalances)) {
        $agentBalanceNumeric = array_sum($agentBasedBalances);
      } elseif (is_numeric($agentBasedBalances)) {
        $agentBalanceNumeric = (float) $agentBasedBalances;
      }

      if (!isset($grouped[$key])) {
        $grouped[$key] = [
          'account_code' => $accountCode,
          'account_name' => $accountName,
          'extra' => $row->account->extra ?? [],
          'credit_limit' => $creditLimit,
          'AgentBasedBalances' => $agentBasedBalances,
          // row_class will be sent to client and used by createdRow
          'row_class' => ($creditLimit !== null && $agentBalanceNumeric !== null && $agentBalanceNumeric > 0 && $creditLimit < $agentBalanceNumeric)
            ? 'table-danger' // change class as you wish
            : '',
        ];
        Log::info($grouped);
      }

      $grouped[$key]["{$currency}_credit"] = $credit;
      $grouped[$key]["{$currency}_debit"] = $debit;

      $totals["{$currency}_credit"] = ($totals["{$currency}_credit"] ?? 0) + $credit;
      $totals["{$currency}_debit"] = ($totals["{$currency}_debit"] ?? 0) + $debit;

      // حوالات غير مستلمة
      if (str_starts_with($accountCode, '13')) {
        $remittanceTotals["{$currency}_credit"] = ($remittanceTotals["{$currency}_credit"] ?? 0) + $credit;
        $remittanceTotals["{$currency}_debit"] = ($remittanceTotals["{$currency}_debit"] ?? 0) + $debit;

        if (!$this->showRemittanceDetails) {
          unset($grouped[$key]);
        }
      }

      // العمولات
      if (str_starts_with($accountCode, '51')) {
        $commissionTotals["{$currency}_credit"] = ($commissionTotals["{$currency}_credit"] ?? 0) + $credit;
        $commissionTotals["{$currency}_debit"] = ($commissionTotals["{$currency}_debit"] ?? 0) + $debit;

        if (!$this->showRemittanceDetails) {
          unset($grouped[$key]);
        }
      }
    }

    // صف المجموع الكلي
    $totalsRow = [
      'account_code' => '',
      'account_name' => 'مجموع الصناديق',
      'extra' => null,
      'AgentBasedBalances' => null,
      'credit_limit' => null
    ];
    foreach ($this->getCurrencyColumns() as $col) {
      $totalsRow[$col] = $totals[$col] ?? 0;
    }

    // صف مجموع حوالات غير مستلمة
    $remittanceRow = null;
    if (!$this->showRemittanceDetails && !empty($remittanceTotals)) {
      $remittanceRow = [
        'account_code' => '13xx',
        'account_name' => 'مجموع الحوالات غير المستلمة',
        'extra' => null,
        'AgentBasedBalances' => null,
        'credit_limit' => null
      ];
      foreach ($this->getCurrencyColumns() as $col) {
        $remittanceRow[$col] = $remittanceTotals[$col] ?? 0;
      }
    }

    $final = [$totalsRow];

    if ($remittanceRow) {
      $final[] = $remittanceRow;
    }
    // صف مجموع العمولات
    if (!$this->showRemittanceDetails && !empty($commissionTotals)) {
      $commissionRow = [
        'account_code' => '51xx',
        'account_name' => 'مجموع العمولات',
        'extra' => null,
        'AgentBasedBalances' => null,
        'credit_limit' => null
      ];
      foreach ($this->getCurrencyColumns() as $col) {
        $commissionRow[$col] = $commissionTotals[$col] ?? 0;
      }

      $final[] = $commissionRow;
    }

    $final = array_merge($final, array_values($grouped));

    $datatable = datatables()->collection(collect($final));

    foreach ($this->getCurrencyColumns() as $column) {
      $datatable->editColumn($column, function ($row) use ($column) {
        return PriceFormat(round($row[$column] ?? 0, 0));
      });
    }
    $datatable->addColumn("actions", function ($row) {
      return view('Balances.actions', compact('row'))->render();
    });
    $datatable->editColumn('AgentBasedBalances', function ($row) {
      $balance = $row['AgentBasedBalances'];

      // Normalize numeric value
      if (is_array($balance)) {
        $balance = array_sum($balance);
      }
      if (!is_numeric($balance)) {
        return $balance; // return as is if it's not numeric
      }

      $absValue =  ($balance);
      $formatted = PriceFormat(round($absValue, 0));

      // Decide color
      if ($balance < 0) {
        return "<span class='text-primary' style='font-weight: bold;'>{$formatted} </span>";
      } else {
        return "<span class='text-danger' style='font-weight: bold;'>{$formatted} </span>";
      }
    });
    $datatable->editColumn('credit_limit', function ($row) {
      $creditLimit = $row['credit_limit'];

      // Normalize numeric value
      if (!is_numeric($creditLimit)) {
        return $creditLimit; // return as is if it's not numeric
      }

      $absValue =  ($creditLimit);
      $formatted = PriceFormat(round($absValue, 0));

      // Decide color
      if ($creditLimit < 0) {
        return "<span style='font-weight: bold;'>{$formatted} </span>";
      } else {
        return "<span   style='font-weight: bold;'>{$formatted} </span>";
      }
    });
    return $datatable->rawColumns(['actions', 'AgentBasedBalances', 'credit_limit'])
      ->setRowId('account_code');
  }

  public function html()
  {
    return $this->builder()
      ->setTableId('balance-table')
      ->addTableClass('table table-bordered')
      ->addTableClass('display cell-border')
      ->columns($this->getColumns())
      ->minifiedAjax()
      ->responsive(true)
      ->autoWidth(false)
      ->dom('Bfrtip')

      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#bebbdfff");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
        'createdRow' => 'function(row, data, dataIndex) {
        if (data.row_class) {
            $(row).addClass(data.row_class);
        }
    }',
      ])
      // ->dom('<"row"' .
      //     '<"col-md-2"<"ms-n2"l>>' .
      //     '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
      //     '>r' . // <-- add this line
      //     't' .
      //     '<"row"' .
      //     '<"col-sm-12 col-md-6"i>' .
      //     '<"col-sm-12 col-md-6"p>' .
      //     '>')
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
    }'
      ])
      ->buttons([
        'colvis',
        [
          'text' => 'تصدير Excel',
          'action' => 'function () {
                        const url = new URL(window.location.href);
                        const details = url.searchParams.get("details") || "1";
                        window.location.href = "/export/balances?details=" + details;
                    }',
        ],
        [
          'extend' => 'csv',
          'text' => 'تصدير CSV',
          'exportOptions' => ['modifier' => ['page' => 'all']],
        ],
      ])
      ->parameters([
        'pageLength' => 20,
        'language' => ['url' => asset('vendor/datatables/ar.json')],

      ]);
  }

  public function getColumns(): array
  {
    $columns = [
      Column::make('account_code')->title('كود الحساب')->addClass('text-center'),
      Column::make('account_name')->title('اسم الحساب')->addClass('text-start'),
      Column::make('credit_limit')->title('حد مديونية')->addClass('text-end'),
      Column::make('AgentBasedBalances')->title('الرصيد المقوم')->addClass('text-start'),

    ];

    $filteredCurrencies = !empty($this->filters['currencies']) && is_array($this->filters['currencies'])
      ? $this->filters['currencies']
      : $this->currencies;
    Log::info($filteredCurrencies);
    foreach ($filteredCurrencies as $currency) {
      $currency_name = Currency::getCurrencyName($currency);
      $columns[] = Column::make("{$currency}_credit")
        ->title("{$currency_name} علينا")
        ->addClass('text-danger text-end fw-bold');

      $columns[] = Column::make("{$currency}_debit")
        ->title("{$currency_name} لنا")
        ->addClass('text-primary text-end fw-bold');
    }
    $columns[] = Column::make("actions")->title('actions')->addClass('text-center');

    return $columns;
  }


  protected function getCurrencyColumns(): array
  {
    $activeCurrencies = !empty($this->filters['currencies']) && is_array($this->filters['currencies'])
      ? $this->filters['currencies']
      : $this->currencies;

    $columns = [];
    foreach ($activeCurrencies as $currency) {
      $columns[] = "{$currency}_credit";
      $columns[] = "{$currency}_debit";
    }

    return $columns;
  }


  protected function filename(): string
  {
    return 'Balance_' . date('YmdHis');
  }
}
