<?php

namespace App\DataTables\Agents;

use App\Models\Currency;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Abivia\Ledger\Messages\Report;
use App\LedgerReports\AccountStatementReport;
use Abivia\Ledger\Http\Controllers\ReportController;
use App\Models\CustomLedgerAccount;
use Yajra\DataTables\CollectionDataTable;
use Yajra\DataTables\EloquentDataTable;

class TransactionsDataTable extends DataTable
{
  public $filters = [];

  public function dataTable($query)
  {


    return (new CollectionDataTable($query))

      ->addColumn("date", function ($data) {

        $id = $data['transfer_id'] ?? null;
        if (!$id) {
          Log::info($data);
          return null;
        }

        $formattedDate = Carbon::parse($data['date'])->format('Y-m-d H:i:s');

        return '<button class="btn btn-sm btn-text-primary" title="' . __('View Transfer') . '"
                x-on:click="$dispatch(\'setTransferId\', { id: ' . $id . ' })">
                ' . $formattedDate . '
            </button>';
      })
      ->addColumn("reference",  function ($data) {
        $number = $data['transfer_number'] ?? '';
        $transfer_side = $data['transfer_side'] ?? '';

        return "<div class='text-center'>
    <span class='fw-bold d-block'>{$number}</span>
    <span class='badge bg-secondary mt-1'>{$transfer_side}</span>
</div>";
      })

      ->addColumn("user", fn($data) => $data['user'])
      ->addColumn("type", fn($data) => $data['transfer_type'] ?? $data['type'])
      ->addColumn("send_fee",  function ($data) {
        $id = $data['transfer_id'] ?? null;
        if (!$id) {

          return null;
        }
        $send_fee = $data['send_fee'] ?? 0;
        return ((float)$send_fee);
      })
      ->addColumn("delivery_fee",  function ($data) {
        $id = $data['transfer_id'] ?? null;
        if (!$id) {
          Log::info($data);
          return null;
        }
        $delivery_fee =  $data['delivery_fee'] ?? 0;
        return ((float)$delivery_fee);
      })

      ->addColumn('debit_amount', function ($data) {
        if ($data['debit'] !== '') {
          return "<span  style='color:blue'>" . PriceFormat($data['debit']) . "</span>";
        }
        return '';
      })
      ->addColumn('credit_amount', function ($data) {
        if ($data['credit'] !== '') {
          return "<span style='color:red'>" . PriceFormat($data['credit']) . "</span>";
        }
        return '';
      })

      ->addColumn('receiver', fn($data) => $data['receiver'] ?? '')
      ->addColumn('description', fn($data) => $data['description'] ?? '')
      ->addColumn('balance', fn($data) => $data['balance'] !== '' ? PriceFormat($data['balance']) : '')
      ->addColumn('review',  function ($data) {
        return view('Statements.actions', compact('data'));
      })
      ->rawColumns(['reference', 'account_name', 'debit_amount', 'credit_amount', 'description', 'balance', 'date', 'delivery_fee', 'send_fee', 'review', "type"]);
  }

  public function query()
  {
    try {
      if (request()->filled('filters')) {
        $this->filters = request()->get('filters');
        Log::info('Filters applied:', $this->filters);
      }
      if (isset($this->filters['account']) && isset($this->filters['currency']))
        $report = \Abivia\Ledger\Messages\Report::fromArray([
          'name' => 'accountStatement',
          'currency' =>  $this->filters['currency'] ?? 'TRY',
          'fromDate' => $this->filters['fromDate'] ?? \Carbon\Carbon::now()->subMonth()->format('Y-m-d'),
          'toDate' => $this->filters['toDate'] ?? \Carbon\Carbon::now()->format('Y-m-d'),
          'options' => [
            'account' => $this->filters['account'] ?? null,
            'depth' => 5,
            'language' => [app()->getLocale()],
          ],
        ]);

      $cacheKey = 'acct_stmt_' . md5(json_encode([$this->filters]));
      //  $reportData = Cache::remember($cacheKey, now()->addMinutes(2), function () use ($report) {
      $reporter = new \Abivia\Ledger\Http\Controllers\ReportController();
      $reportData = $reporter->generate($report);
      // });

      return collect($reportData);
    } catch (\Exception $e) {
      Log::error($e);
      Log::error('Error generating account statement report: ' . $e->getMessage());
      return collect([]);
    }
  }
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->addTableClass('table table-bordered   display cell-border')

      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'rowCallback' => 'function(row, data, index) {
        if (index % 2 === 0) {
            $(row).css("background-color", "#f9f9f9");
        } else {
            $(row).css("background-color", "#ffffff");
        }
    }',
        'createdRow' => 'function(row, data, dataIndex) {
        if (data.description === "' . __('إجمالي الحركة') . '") {
            $(row).addClass("table-warning fw-bold");
        }
    }',
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',

      ])
      ->setTableId('transaction-table')
      ->columns($this->getColumns())
      ->minifiedAjax()
      ->processing(true)
      ->serverSide(true)
      ->ajax([
        'url' => '',
        'type' => 'GET',
        'data' => 'function(data) {
                    data.filters = window.filters || {};
                }',
      ])
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>r' .
        't' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->lengthMenu([[1000, 1500, 2000, 2500], [1000, 1500, 2000, 2500]])
      ->orderBy([0, "desc"])


      ->buttons([]);
  }

  public function getColumns(): array
  {
    return [
      Column::make('#')->title(__('#')),
      Column::make('date')->title(__('statements.date')),
      Column::make('type')->title(__('statements.type')),
      Column::make('reference')->title(__('statements.reference')),


      Column::make('description')->title(__('statements.description')),

      Column::make('debit_amount')->title(__('statements.debit')),
      Column::make('credit_amount')->title(__('statements.credit')),
      Column::make('balance')->title(__('statements.balance')),
      Column::make('review')->title(__('statements.review')),
    ];
  }



  /**
   * Get columns based on current filters
   */
  public function getColumnsForFilters(array $filters = []): array
  {
    $this->filters = array_merge($this->filters, $filters);
    return $this->buildColumns();
  }

  protected function filename(): string
  {
    return 'Transaction_' . date('YmdHis');
  }
}
