<?php

namespace App\DataTables;

use App\Models\Agent;
use App\Models\Currency;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class CreditSettingDataTable extends DataTable
{
  protected $currencies;
  public function __construct()
  {
    $this->currencies = Currency::get();
  }
  /**
   * Build the DataTable class.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return datatables()
      ->eloquent($query)
      ->addColumn('currencies', function (Agent $agent) {
        $type = "currency";
        return view('Agents.partials.currency_settings', [
          'type' => $type,
          'agent' => $agent,
          'currencies' => $this->currencies, // optional: add your own order logic
        ], compact('type'))->render();
      })
      ->editColumn('name', function (Agent $agent) {
        return  getAgentHtml($agent);
      })
      ->filter(function ($query) {
        if (request()->has('search') && !empty(request('search')['value'])) {
          $keyword = request('search')['value'];

          $query->where(function ($q) use ($keyword) {
            $q->where('agents.name', 'like', "%{$keyword}%")
              ->orWhere('agents.code', 'like', "%{$keyword}%")
              ->orWhereHas('currencyVisibilities', function ($q2) use ($keyword) {
                $q2->where('currency_code', 'like', "%{$keyword}%")
                  ->orWhereHas('currency', function ($q5) use ($keyword) {
                    $q5->where('name', 'like', "%{$keyword}%");
                  });
              })
              ->orWhereHas('GlobalCreditLimit', function ($q3) use ($keyword) {
                $q3->where('base_currency_code', 'like', "%{$keyword}%")
                  ->orWhere('base_credit_limit', 'like', "%{$keyword}%")
                  ->orWhere('approval_credit_limit', 'like', "%{$keyword}%");
              })
              ->orWhereHas('CurrencyCreditLimit', function ($q4) use ($keyword) {
                $q4->where('currency_code', 'like', "%{$keyword}%")
                  ->orWhere('credit_limit', 'like', "%{$keyword}%");
              });
          });
        }
      })

      ->rawColumns(['global_credit', 'currencies', 'send_transfer', 'send_approval', 'name']);
  }
  /**
   * Get the query source of dataTable.
   */
  public function query(Agent $model): QueryBuilder
  {
    return $model->newQuery()->where('type', "!=", 'cashier')->IsActive()
      ->with(['currencyVisibilities.currency', 'GlobalCreditLimit', 'CurrencyCreditLimit']);
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
        'dom' => '<"row"<"col-sm-12 col-md-6"B><"col-sm-12 col-md-6"f>>' .
          '<"row"<"col-sm-12"tr>>' .
          '<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
        'buttons' => [
          [
            'text' => '<i class="ti ti-plus"></i> ' . __("Apply On All"),
            'className' => 'btn btn-primary btn-sm',
            'action' => 'function() {
              if (typeof showCreditSettingsModal === "function") {
                showCreditSettingsModal();
              } else {
                console.error("showCreditSettingsModal function not found");
                // Fallback: try to show modal directly
                const modal = document.getElementById("creditSettingsModal");
                if (modal) {
                  const bsModal = new bootstrap.Modal(modal);
                  bsModal.show();
                } else {
                  alert("Credit settings modal not found. Please refresh the page.");
                }
              }
            }'
          ]
        ],
        'initComplete' => 'function() {
          // Define the function globally when DataTable is ready
          window.showCreditSettingsModal = function() {
            const modal = document.getElementById("creditSettingsModal");
            if (modal) {
              const bsModal = new bootstrap.Modal(modal);
              bsModal.show();
            } else {
              console.error("Credit settings modal not found");
              alert("Credit settings modal not found. Please refresh the page.");
            }
          };

          console.log("showCreditSettingsModal function defined globally");
        }'
      ])
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>r' . // <-- add this line
        't' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->setTableId('creditsetting-table')
      ->columns($this->getColumns())
      ->minifiedAjax()
      ->orderBy(1);
  }

  /**
   * Define the columns.
   */
  public function getColumns(): array
  {
    return [
      Column::make('id')->title('ID'),
      Column::make('name')->title('اسم العميل'),

      Column::computed('currencies')
        ->title('الاعدادات')
        ->exportable(false)
        ->printable(false)
        ->addClass('text-left'),
    ];
  }


  /**
   * Export filename.
   */
  protected function filename(): string
  {
    return 'CreditSetting_' . date('YmdHis');
  }
}
