<?php

namespace App\DataTables;

use App\Models\Exchange;
use App\Models\LedgerTransfer;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class ExchangeDataTable extends DataTable
{
  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('action', function ($data) {
        return view('Exchange.actions', compact('data'));
      })
      ->addColumn("amount", function ($query) {
        return PriceFormat($query->amount, $query->currency);
      })
      ->addColumn("delivery_amount", function ($query) {
        return PriceFormat($query->delivery_amount, $query->delivery_currency);
      })

      ->addColumn("agent", function ($query) {
        return $query->sender->name ?? '';
      })
      ->addColumn("created_by", function ($query) {
        return $query->user->name ?? 'N/A';
      })
      ->addColumn("created_at", function ($query) {
        return $query->created_at->format('Y-m-d H:i');
      })
      ->editColumn("note", function ($query) {
        return $query->notes ?? __("general.no_note");;
      })
      ->rawColumns(['action', 'amount', 'delivery_amount', 'created_at', 'created_by', 'note'])
      ->setRowId('id');
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(LedgerTransfer $model): QueryBuilder
  {
    $query = $model->newQuery()->exchange()->orderby('id', 'desc');
    if (auth()->guard('agent')->check()) {
      $query->where('sender_id', auth()->guard('agent')->user()->agent->rootAgent()->id);
    }
    return $query;
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('exchange-table')
      ->columns($this->getColumns())
      ->minifiedAjax()
      //->dom('Bfrtip')
      ->orderBy(1)
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ])
      ->buttons([]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    return [
      Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center'),
      Column::make('reference')
        ->title(__('general.reference')),
      Column::make('created_at')

        ->title(__('general.created_at')),
      Column::make('created_by')
        ->title(__('general.created_by')),
      Column::make('note')
        ->title(__('general.note')),

      Column::make('amount')
        ->title(__('general.from_amount'))
        ->addClass('text-end'),
      Column::make('delivery_amount')
        ->title(__('general.to_amount'))
        ->addClass('text-end'),
      Column::make('agent')
        ->title(__('general.agent'))
        ->width(100),





    ];
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'Exchange_' . date('YmdHis');
  }
}
