<?php

namespace App\DataTables;

use App\Models\CustomLedgerAccount;
use App\Models\Currency;
use App\Models\BalanceOverwrite;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Illuminate\Support\Facades\DB;

class TransferProfitDataTable extends DataTable
{
  protected $currencies;

  public function __construct()
  {
    $this->currencies = Currency::orderBy('order_id')->get();
  }

  /**
   * Build the DataTable class.
   *
   * @param QueryBuilder $query Results from query() method.
   */
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    $dataTable = (new EloquentDataTable($query))
      ->addColumn('action', function ($account) {
        return view('transferprofit.action', compact('account'));
      })
      ->addColumn('account_name', function ($account) {
        // Try to get name in current locale, fallback to any available name, then to code
        $name = $account->names->firstWhere('language', app()->getLocale())?->name
          ?? $account->names->first()?->name
          ?? $account->code;
        return $name . ' - ' . $account->code;
      })
      ->addColumn('account_code', function ($account) {
        return $account->code;
      });

    // Add individual currency balance columns
    foreach ($this->currencies as $currency) {
      $dataTable->addColumn("balance_{$currency->code}", function ($account) use ($currency) {
        $balance = BalanceOverwrite::where('ledgerUuid', $account->ledgerUuid)
          ->where('currency', $currency->code)
          ->first();

        if ($balance && (float) $balance->balance != 0) {
          $amount = (float) $balance->balance;
          $formatted = number_format(abs($amount), 2);
          $displayValue = $amount < 0 ? $formatted . '-' : $formatted;

          // Add CSS class based on balance
          $cssClass = $amount > 0 ? 'balance-positive' : 'balance-negative';
          return '<span class="' . $cssClass . '">' . $displayValue . '</span>';
        }

        return '<span class="balance-zero">0</span>';
      });
    }

    // Get all currency column names for rawColumns
    $currencyColumns = $this->currencies->map(function ($currency) {
      return "balance_{$currency->code}";
    })->toArray();

    return $dataTable->setRowId('ledgerUuid')->rawColumns(array_merge(['action'], $currencyColumns));
  }

  /**
   * Get the query source of dataTable.
   */
  public function query(CustomLedgerAccount $model): QueryBuilder
  {
    return $model->newQuery()
      ->with(['names'])
      ->where('code', 'like', '4%')
      ->where('category', false) // Only get actual accounts, not categories
      ->orderBy('code');
  }

  /**
   * Optional method if you want to use the html builder.
   */
  public function html(): HtmlBuilder
  {
    $columns = $this->getColumns();

    return $this->builder()
      ->setTableId('transferprofit-table')
      ->columns($columns)
      ->minifiedAjax()
      ->orderBy(1)
      ->selectStyleSingle()
      ->buttons([])
      ->parameters([
        'language' => [
          'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
        ],
        'processing' => true,
        'serverSide' => true,

        'deferRender' => true,
        'scrollX' => true,
        'pageLength' => 10, // Reduce default page length
        'lengthMenu' => [10, 25, 50, 100],
        'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
      ]);
  }

  /**
   * Get the dataTable columns definition.
   */
  public function getColumns(): array
  {
    $columns = [
      Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(80)
        ->addClass('text-center')
        ->title('ترحيل'),
      Column::make('account_name')
        ->title('اسم الحساب')
        ->searchable(true)
        ->orderable(true)
        ->addClass('text-center'),
    ];

    // Add currency columns dynamically
    foreach ($this->currencies as $currency) {
      $columns[] = Column::make("balance_{$currency->code}")
        ->title($currency->getTranslation('name', app()->getLocale()) ?? $currency->code)
        ->searchable(false)
        ->orderable(false)
        ->addClass('text-center');
    }

    return $columns;
  }

  /**
   * Get the filename for export.
   */
  protected function filename(): string
  {
    return 'TransferProfit_' . date('YmdHis');
  }
}
