<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class RedisSessionHelper
{
  /**
   * Clear all Redis data for a specific user
   */
  public static function clearUserSessions($userId): void
  {
    try {
      // Get Redis configuration
      $sessionPrefix = config('session.prefix', 'laravel_session:');
      $cachePrefix = config('cache.prefix', 'laravel_cache:');
      $databasePrefix = config('database.redis.default.prefix', '');

      // Clear user sessions from Redis
      self::clearRedisSessions($userId, $sessionPrefix, $databasePrefix);

      // Clear user cache from Redis
      self::clearRedisCache($userId, $cachePrefix, $databasePrefix);

      // Clear any other user-specific Redis keys
      self::clearUserSpecificKeys($userId, $databasePrefix);

      Log::info("Redis data cleared for user ID: {$userId}");
    } catch (\Exception $e) {
      Log::error("Error clearing Redis data for user ID {$userId}: " . $e->getMessage());
    }
  }

  /**
   * Clear Redis sessions for a user
   */
  private static function clearRedisSessions($userId, $sessionPrefix, $databasePrefix): void
  {
    try {
      // Clear user session tracking with prefixes
      $userSessionKeys = [
        "user_sessions:{$userId}",
        "{$databasePrefix}user_sessions:{$userId}",
        "user_session_{$userId}",
        "{$databasePrefix}user_session_{$userId}"
      ];

      foreach ($userSessionKeys as $key) {
        Redis::del($key);
      }

      // Clear pattern-based user sessions with prefixes
      $userSessionPatterns = [
        "*user_session_{$userId}_*",
        "*user_sessions:{$userId}*",
        "*session*user*{$userId}*"
      ];

      foreach ($userSessionPatterns as $pattern) {
        $userSessions = Redis::keys($pattern);
        if (!empty($userSessions)) {
          Redis::del($userSessions);
        }
      }

      // Clear Laravel session files if they exist
      $sessionPatterns = [
        "*{$sessionPrefix}*",
        "*session*"
      ];

      foreach ($sessionPatterns as $sessionPattern) {
        $allSessions = Redis::keys($sessionPattern);

        foreach ($allSessions as $sessionKey) {
          $sessionData = Redis::get($sessionKey);
          if ($sessionData && (
            strpos($sessionData, '"user_id":' . $userId) !== false ||
            strpos($sessionData, '"id":' . $userId) !== false ||
            strpos($sessionData, '"user":' . $userId) !== false
          )) {
            Redis::del($sessionKey);
          }
        }
      }

      // Clear agent-specific sessions with prefixes
      $agentSessionPatterns = [
        "*agent_session_{$userId}_*",
        "*agent*user*{$userId}*",
        "*session*agent*{$userId}*"
      ];

      foreach ($agentSessionPatterns as $pattern) {
        $agentSessions = Redis::keys($pattern);
        if (!empty($agentSessions)) {
          Redis::del($agentSessions);
        }
      }
    } catch (\Exception $e) {
      Log::error("Error clearing Redis sessions for user ID {$userId}: " . $e->getMessage());
    }
  }

  /**
   * Clear Redis cache for a user
   */
  private static function clearRedisCache($userId, $cachePrefix, $databasePrefix): void
  {
    try {
      // Clear user-specific cache keys with prefixes
      $userCacheKeys = [
        "user_{$userId}_permissions",
        "user_{$userId}_settings",
        "user_{$userId}_preferences",
        "user_{$userId}_notifications",
        "user_{$userId}_profile",
        "user_{$userId}_roles",
        "{$databasePrefix}user_{$userId}_permissions",
        "{$databasePrefix}user_{$userId}_settings",
        "{$databasePrefix}user_{$userId}_preferences",
        "{$databasePrefix}user_{$userId}_notifications",
        "{$databasePrefix}user_{$userId}_profile",
        "{$databasePrefix}user_{$userId}_roles"
      ];

      foreach ($userCacheKeys as $key) {
        Redis::del($key);
      }

      // Clear pattern-based cache with prefixes
      $userCachePatterns = [
        "*user_{$userId}_*",
        "*user*{$userId}*"
      ];

      foreach ($userCachePatterns as $pattern) {
        $userCacheKeys = Redis::keys($pattern);
        if (!empty($userCacheKeys)) {
          Redis::del($userCacheKeys);
        }
      }

      // Clear agent-specific cache with prefixes
      $agentCachePatterns = [
        "*agent_{$userId}_*",
        "*agent*{$userId}*"
      ];

      foreach ($agentCachePatterns as $pattern) {
        $agentCacheKeys = Redis::keys($pattern);
        if (!empty($agentCacheKeys)) {
          Redis::del($agentCacheKeys);
        }
      }
    } catch (\Exception $e) {
      Log::error("Error clearing Redis cache for user ID {$userId}: " . $e->getMessage());
    }
  }

  /**
   * Clear any other user-specific Redis keys
   */
  private static function clearUserSpecificKeys($userId, $databasePrefix): void
  {
    try {
      // Clear user activity tracking with prefixes
      $activityKeys = [
        "user_activity:{$userId}",
        "user_last_seen:{$userId}",
        "user_login_attempts:{$userId}",
        "agent_activity:{$userId}",
        "agent_last_seen:{$userId}",
        "agent_login_attempts:{$userId}",
        "{$databasePrefix}user_activity:{$userId}",
        "{$databasePrefix}user_last_seen:{$userId}",
        "{$databasePrefix}user_login_attempts:{$userId}",
        "{$databasePrefix}agent_activity:{$userId}",
        "{$databasePrefix}agent_last_seen:{$userId}",
        "{$databasePrefix}agent_login_attempts:{$userId}"
      ];

      foreach ($activityKeys as $key) {
        Redis::del($key);
      }

      // Clear any other patterns you might have with prefixes
      $otherPatterns = [
        "*user_{$userId}_*",
        "*session_user_{$userId}_*",
        "*auth_user_{$userId}_*",
        "*agent_{$userId}_*",
        "*session_agent_{$userId}_*",
        "*auth_agent_{$userId}_*",
        "*user*{$userId}*",
        "*agent*{$userId}*",
        "*session*{$userId}*",
        "*auth*{$userId}*"
      ];

      foreach ($otherPatterns as $pattern) {
        $keys = Redis::keys($pattern);
        if (!empty($keys)) {
          Redis::del($keys);
        }
      }
    } catch (\Exception $e) {
      Log::error("Error clearing user-specific Redis keys for user ID {$userId}: " . $e->getMessage());
    }
  }

  /**
   * Force logout user from all sessions (Redis + Cache)
   */
  public static function forceLogoutUser($userId, $guard = 'web'): void
  {
    try {
      // Clear Redis sessions
      self::clearUserSessions($userId);

      // Clear file cache if any
      Cache::forget("user_session_{$userId}");
      Cache::forget("agent_session_{$userId}");

      // Clear any other cache keys
      $cacheKeys = [
        "user_{$userId}_permissions",
        "user_{$userId}_settings",
        "user_{$userId}_preferences",
        "user_{$userId}_notifications",
        "agent_{$userId}_permissions",
        "agent_{$userId}_settings"
      ];

      foreach ($cacheKeys as $key) {
        Cache::forget($key);
      }

      Log::info("Force logout completed for user ID: {$userId} (Guard: {$guard})");
    } catch (\Exception $e) {
      Log::error("Error in force logout for user ID {$userId}: " . $e->getMessage());
    }
  }
}
