<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    public function showLoginForm()
    {
        return view('AgentPanel.auth.login');
    }

    public function login(Request $request)
    {
        // Throttle login attempts
        $this->checkTooManyFailedAttempts($request);

        config(['fortify.guard' => 'agent']);
        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);


        $username = $request->input('email');
        $check_status = User::where('email', $username)->first();
        if ($check_status && $check_status->status === 0) {
            return back()->withErrors([
                'email' => __('Your account is disabled.'),
            ]);
        }
        if (Auth::guard('agent')->attempt($credentials, $request->boolean('remember'))) {
            // Clear login attempts when successful
            RateLimiter::clear($this->throttleKey($request));

            config(['session.cookie' => 'agent_session']);
            $request->session()->regenerate();
            return redirect()->intended(route('agent.dashboard'));
        }

        // Increment failed attempts
        RateLimiter::hit($this->throttleKey($request));

        return back()->withErrors([
            'email' => __('invalid credentials'),
        ]);
    }

    public function logout(Request $request)
    {
        Auth::guard('agent')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login');
    }

    /**
     * Get the throttle key for the given request.
     */
    protected function throttleKey(Request $request): string
    {
        return Str::transliterate(Str::lower($request->input('email')) . '|' . $request->ip());
    }

    /**
     * Ensure the login request is not rate limited.
     */
    protected function checkTooManyFailedAttempts(Request $request): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey($request), 3)) {
            return;
        }

        $seconds = RateLimiter::availableIn($this->throttleKey($request));
        $username = $request->input('email');
        User::where('email', $username)->update(['status' => 0]);
        throw ValidationException::withMessages([
            'email' => trans('throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }
    public  function showChangePasswordForm()
    {
        return view('AgentPanel.auth.change');
    }
}
