<?php

namespace App\Http\Controllers\Demo;

use App\Http\Controllers\Controller;
use App\Helpers\TransferValidationHelper;
use App\Rules\MinWords;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TransferValidationController extends Controller
{
  /**
   * Show demo form for transfer validation
   */
  public function showDemo()
  {
    $beneficiaryRules = TransferValidationHelper::getBeneficiaryNameRules();
    $phoneRules = TransferValidationHelper::getPhoneNumberRules();

    return view('demo.transfer-validation', compact('beneficiaryRules', 'phoneRules'));
  }

  /**
   * Process demo form submission
   */
  public function processDemo(Request $request)
  {
    // Get validation rules from settings
    $rules = TransferValidationHelper::getLaravelValidationRules();
    $messages = TransferValidationHelper::getLaravelValidationMessages();

    // Add custom MinWords rule for beneficiary name
    if (isset($rules['beneficiary_name'])) {
      $beneficiaryRules = TransferValidationHelper::getBeneficiaryNameRules();
      $rules['beneficiary_name'] = str_replace(
        "|min_words:{$beneficiaryRules['min_words']},{$beneficiaryRules['max_words']}",
        '',
        $rules['beneficiary_name']
      );
    }

    $validator = Validator::make($request->all(), $rules, $messages);

    // Add custom MinWords rule
    if (isset($rules['beneficiary_name'])) {
      $beneficiaryRules = TransferValidationHelper::getBeneficiaryNameRules();
      $validator->addRules([
        'beneficiary_name' => [new MinWords($beneficiaryRules['min_words'], $beneficiaryRules['max_words'])]
      ]);
    }

    if ($validator->fails()) {
      return response()->json([
        'success' => false,
        'errors' => $validator->errors()
      ], 422);
    }

    // If validation passes, you can process the transfer
    return response()->json([
      'success' => true,
      'message' => 'تم التحقق من البيانات بنجاح',
      'data' => [
        'beneficiary_name' => $request->beneficiary_name,
        'phone_number' => $request->phone_number,
        'validation_rules_applied' => $rules
      ]
    ]);
  }

  /**
   * API endpoint for validating beneficiary name
   */
  public function validateBeneficiaryName(Request $request)
  {
    $request->validate([
      'name' => 'required|string'
    ]);

    $result = TransferValidationHelper::validateBeneficiaryName($request->name);

    return response()->json($result);
  }

  /**
   * API endpoint for validating phone number
   */
  public function validatePhoneNumber(Request $request)
  {
    $request->validate([
      'phone' => 'required|string'
    ]);

    $result = TransferValidationHelper::validatePhoneNumber($request->phone);

    return response()->json($result);
  }

  /**
   * Get current validation settings
   */
  public function getSettings()
  {
    return response()->json([
      'beneficiary_name' => TransferValidationHelper::getBeneficiaryNameRules(),
      'phone_number' => TransferValidationHelper::getPhoneNumberRules(),
      'laravel_rules' => TransferValidationHelper::getLaravelValidationRules(),
      'laravel_messages' => TransferValidationHelper::getLaravelValidationMessages(),
    ]);
  }
}
