<?php

namespace App\Http\Controllers;

use App\Services\TrialBalanceService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class TrialBalanceController extends Controller
{
  protected $trialBalanceService;

  public function __construct(TrialBalanceService $trialBalanceService)
  {
    $this->trialBalanceService = $trialBalanceService;
  }

  /**
   * Display the trial balance page
   */
  public function index()
  {
    return view('trial-balance.index');
  }

  /**
   * Get trial balance data via AJAX
   */
  public function getTrialBalance(Request $request): JsonResponse
  {
    $request->validate([
      'currency' => 'required|string',
      'depth' => 'required|integer|min:1|max:5',
      'to_date' => 'required|date',
      'domain_uuid' => 'nullable|string',
      'include_zero_balances' => 'boolean'
    ]);

    try {
      $data = $this->trialBalanceService->generateTrialBalance(
        $request->currency,
        $request->depth,
        null, // fromDate not needed for Abivia API
        $request->to_date,
        $request->domain_uuid,
        $request->include_zero_balances ?? false
      );

      return response()->json([
        'success' => true,
        'data' => $data,
        'summary' => $this->calculateSummary($data),
        'generated_at' => now()->format('Y-m-d H:i:s')
      ]);
    } catch (\Exception $e) {
      return response()->json([
        'success' => false,
        'message' => 'حدث خطأ أثناء إنشاء الميزانية العمومية: ' . $e->getMessage()
      ], 500);
    }
  }

  /**
   * Export trial balance to Excel
   */
  public function export(Request $request)
  {
    $request->validate([
      'currency' => 'required|string',
      'depth' => 'required|integer',
      'to_date' => 'required|date',
      'domain_uuid' => 'nullable|string',
      'include_zero_balances' => 'boolean'
    ]);

    try {
      $data = $this->trialBalanceService->generateTrialBalance(
        $request->currency,
        $request->depth,
        null, // fromDate not needed for Abivia API
        $request->to_date,
        $request->domain_uuid,
        $request->include_zero_balances ?? false
      );

      $filename = "trial_balance_{$request->currency}_to_{$request->to_date}.xlsx";

      return $this->trialBalanceService->exportToExcel($data, $filename);
    } catch (\Exception $e) {
      return back()->withErrors(['error' => 'حدث خطأ أثناء التصدير: ' . $e->getMessage()]);
    }
  }

  /**
   * Calculate summary statistics
   */
  private function calculateSummary($data)
  {
    $totalBalance = 0;
    $accountCount = count($data);

    foreach ($data as $account) {
      $totalBalance += $account['total_balance'] ?? 0;
    }

    return [
      'total_accounts' => $accountCount,
      'total_balance' => $totalBalance,
      'is_balanced' => abs($totalBalance) < 0.01
    ];
  }
}
