<?php

namespace App\Imports;

use App\Models\Portfolio;
use App\Models\Property;
use App\Models\User;
use Illuminate\Validation\Rule;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\{
    ToModel,
    WithHeadingRow,
    WithValidation,
    SkipsErrors,
    SkipsFailures,
    WithBatchInserts,
    WithChunkReading,
    Importable
};
use Maatwebsite\Excel\Validators\Failure;
use Throwable;

class PropertiesImport implements
    ToModel,
    WithHeadingRow,
    WithChunkReading
{
    use Importable;

    public $failures = [];
    public $errors = [];

    private $portfolios;
    private $tenants;
    private $qualificationMap = [];

    public function __construct()
    {
        $this->portfolios = Portfolio::all(['id', 'name'])->pluck('id', 'name');
        $this->tenants = User::role('Tenant')->get(['id', 'name'])->pluck('id', 'name');

        $property = new Property();
        foreach ($property->qualifications['qualifications'] as $key => $data) {
            $this->qualificationMap[strtolower($data['name']['tr'])] = $key;
            $this->qualificationMap[strtolower($data['name']['en'])] = $key;
        }
    }
    public function model(array $row)
    {
        Log::info($row);

        return new Property([
            'portfolio_id' => $this->portfolios[$row['portfolio']] ?? 1,
            'independent_unit_no' => $row['bbno'] ?? null,
            'qualification' => $this->mapQualification($row['nitelik'] ?? null),
            'type' => $row['tip'],
            'usage_type' => $row['kullanim_turu'],
            'net_sqm' => $row['netm2'] ?? null,
            'gross_sqm' => $row['brut2'] ?? null,
            'ada' => $row['ada'] ?? null,
            'parcel' => $row['parcel'] ?? null,
            'sheet' => $row['sheet'] ?? null,
            'is_active' => in_array($row['durum'] ?? 1, ['Aktif', 'Active', 1]) ? true : false,
        ]);
        if (!empty($row['contract_start']) && !empty($row['contract_end']) && !empty($row['tenant'])) {
            $tenantId = $this->tenants[$row['tenant']] ?? null;

            if ($tenantId) {
                \App\Models\Contract::create([
                    'property_id'    => $property->id,
                    'tenant_id'      => $tenantId,
                    'contract_type'  => 'Tenant',
                    'contract_start' => $row['contract_start'],
                    'contract_end'   => $row['contract_end'],
                    'rent_start'     => $row['contract_start'],
                    'status'         => 1,
                ]);
            }
        }
    }
    private function mapQualification(?string $input): ?string
    {
        if (!$input) return null;

        $normalized = strtolower(trim($input));
        return $this->qualificationMap[$normalized] ?? null;
    }
    public function rules(): array
    {
        return [
            '*.name' => ['required', 'string', 'max:255'],
            '*.address' => ['required', 'string'],
            '*.city' => ['required', 'string'],
            '*.owner_email' => ['nullable', 'email'],
        ];
    }

    public function onFailure(Failure ...$failures)
    {
        $this->failures = $failures;
    }

    public function onError(Throwable $e)
    {
        Log::info($e);
        $this->errors[] = $e->getMessage();
    }

    public function batchSize(): int
    {
        return 1000;
    }

    public function chunkSize(): int
    {
        return 1000;
    }
}
