<?php

namespace App\Livewire\AgentPanel\Transfer;

use Livewire\Component;
use App\Exceptions\InvalidExchangeRateException;
use App\Models\Agent;
use App\Models\Currency;
use App\Models\Exchange;
use App\Models\LedgerExchange;
use App\Models\LedgerTransfer;
use App\Services\Transfer\ExchangeService;
use Illuminate\Support\Facades\Log;

use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\ExchangeRates\ExchangeRateManager;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Events\AuditCustom;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Gate;

class AddExchange extends Component
{
    public $exchange = ['currency' => "USD", 'delivery_currency' => "USD", 'rate' => 1,   'amount' => null, 'delivery_amount' => null, 'type' => "exchange", 'rate_factor' => "multiply"];
    public $agents = [];
    public  $currencies = [];
    public string $createExchangeMessage = '';
    public bool $canCreateExchange = false;
    public $send_flag = false;
    public $exchange_currency;

    protected  function  getListeners()
    {
        return [
            'SetExchangeAgent' => 'SetExchangeAgent',
            'SetExchange' => 'setExchange',
            "SendExchange" => "save"
        ];
    }

    public function updatedExchangeCurrency($value)
    {
        // Example value: "USD_EUR"
        if (strpos($value, '_') !== false) {
            [$from, $to] = explode('_', $value);
            $this->exchange['currency'] = $from;
            $this->exchange['delivery_currency'] = $to;
        }
    }

    public function setExchange($data)
    {
        Log::info("Setting exchange with data: ", $data);

        $this->exchange['currency'] = $data['from'];
        $this->exchange['delivery_currency'] = $data['to'];
        $this->exchange['rate'] = $data['rate'];
        $this->exchange['rate_factor'] = $data['factor'] == "/" ? "divide" : "multiply";

        $this->Calculate();
    }
    // public function  SetExchangeAgent($data)
    // {


    //     $this->exchange['sender_id'] = (int) $data['agent_id'];
    //     $this->exchange['receiver_id'] = (int) $data['agent_id'];
    // }
    public function mount()
    {
        $this->agents = LedgerExchange::GetSourceAgents();;
        $this->currencies = Currency::get();
        $this->exchange['currency'] = $this->currencies[0]->code ?? "USD";
        $this->exchange['from_currency'] = $this->currencies->where("is_default", false)->first()->code ?? "USD";
        $this->exchange['to_currency'] = $this->currencies->where("is_default", true)->first()->code ?? "USD";
        $response = Gate::inspect('create', \App\Models\LedgerExchange::class);

        $this->canCreateExchange = $response->allowed();
        $this->createExchangeMessage = $response->message() ?? '';
        $this->exchange['sender_id'] = auth()->guard('agent')->user()->agent->id ?? 1;
        $this->exchange['receiver_id'] = auth()->guard('agent')->user()->agent->id ?? 1;
    }



    public function updated()
    {
        if (isset($this->exchange['currency']) && isset($this->exchange['delivery_currency'])) {
            $this->exchange['rate_factor'] = getCurrencyFactor($this->exchange['currency'], $this->exchange['delivery_currency'], 'balance');
        }

        $this->Calculate();
    }
    public function Calculate()
    {
        $rate = exchange_rate($this->exchange['currency'], $this->exchange['delivery_currency'], 'transfer');

        if (!$rate) {

            $this->send_flag = false;
            return;
        }
        $this->send_flag = true;
        $this->exchange['rate'] = $rate;
        $from_amount = $this->exchange['amount'] ?? 0;
        $from_currency = $this->exchange['delivery_currency'] ?? "EUR";
        if (!$from_amount || !$from_currency || !$this->exchange['delivery_currency']) {

            return;
        }

        $converter = new CurrencyConverter();
        $to_amount = $converter->convert($from_amount, $from_currency,  $this->exchange['currency'],  $this->currencies, 'transfer', $this->exchange['rate'], $this->exchange['rate_factor'] == 'multiply' ? '*' : '/');


        $this->exchange['delivery_amount'] = $to_amount;
    }



    public  function ConfirmTransfer()
    {
        if (!isset($this->exchange['sender_id'])) {
            $this->dispatch("sweetalert:error", ['msg' => __('exchange.select_sender'), 'title' => 'Error']);
            return;
        }
        try {
            $rules = [
                'exchange.currency' => 'required|different:exchange.delivery_currency',
                'exchange.delivery_currency' => 'required|different:exchange.currency',

                'exchange.receiver_id' => 'nullable',

                'exchange.amount' => 'required|numeric|min:0.01',
                'exchange.delivery_amount' => 'required|numeric|min:0.01',

            ];
            $this->validate($rules);
            $currency_word = $this->getAmountInWords($this->exchange['amount'], $this->exchange['currency']);
            $this->dispatch("areyousure", ['msg' => __("alerts.areyousure"), 'emitName' => 'SendExchange', 'action' => __("YES"), 'title' => __("alerts.areyousure_exchange_transfer", ["currency_word" => $currency_word]), 'data' => []]);
        } catch (InvalidExchangeRateException $e) {
            DB::rollBack();
            Log::warning("Exchange failed due to invalid rate: " . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => $e->getMessage(),
                'title' => 'Invalid Rate'
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Exchange Save Failed: ' . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => 'Exchange failed: ' . $e->getMessage(),
                'title' => 'Error'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Exchange Save Failed: ' . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => 'Exchange failed: ' . $e->getMessage(),
                'title' => 'Error'
            ]);
        }
    }

    public function save()
    {


        $this->exchange['reference'] = generateReferenceNumber("EXG");

        try {
            DB::beginTransaction();
            LedgerTransfer::disableAuditing();
            Log::info("Creating exchange");
            $ex = LedgerTransfer::create($this->exchange);

            Log::info("Calling ExchangeService");
            $service = new ExchangeService();
            $service->InsertExchangeEntries($ex);

            DB::commit();

            LedgerTransfer::enableAuditing();
            $ex->auditEvent = 'Create_Exchange';
            $ex->isCustomEvent = true;
            $ex->auditCustomOld = [];

            $ex->auditCustomNew = [
                "sender" => $ex->sender->name ?? "",
                "receiver" => $ex->receiver->name ?? "",
                "amount" => $ex->amount,
                "receiver_fee" => $ex->receiver_fee,
                "sender_fee" => $ex->sender_fee,
                "delivery_amount" => $ex->delivery_amount,
                "currency" => $ex->currency,
                "delivery_currency" => $ex->delivery_currency,
                "status" => "Completed"
            ];

            // Step 6: Dispatch the custom audit event
            Event::dispatch(new AuditCustom($ex));
            $ex->disableAuditing();

            $this->dispatch("RefreshYajraDatatable", ['table' => 'exchange-table']);
            $this->dispatch("sweetalert:success", [
                'msg' => __('exchange.exchange_added_successfully'),
                'title' => 'Success'
            ]);
        } catch (InvalidExchangeRateException $e) {
            DB::rollBack();
            Log::warning("Exchange failed due to invalid rate: " . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => $e->getMessage(),
                'title' => 'Invalid Rate'
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Exchange Save Failed: ' . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => 'Exchange failed: ' . $e->getMessage(),
                'title' => 'Error'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Exchange Save Failed: ' . $e->getMessage());

            $this->dispatch("sweetalert:error", [
                'msg' => 'Exchange failed: ' . $e->getMessage(),
                'title' => 'Error'
            ]);
        }
    }


    public function getAmountInWords($amount, $currency)
    {
        if ($amount && $currency) {
            return toCurrencyWords(
                floatval($amount),
                $currency,
                app()->getLocale()
            );
        }
        return '';
    }
    public function render()
    {
        return view('livewire.agent-panel.transfer.add-exchange');
    }
}
