<?php

namespace App\Livewire\Agents;

use App\Events\AgentForceLogout;
use App\Exceptions\AgentHasBalanceException;
use App\Models\Agent;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Livewire\Component;
use PeterColes\Countries\CountriesFacade;

class Add extends Component
{
  public $agents = [];
  public $types = ['agent', 'account', 'cashier'];
  public $countries;
  public $agent = ['type' => 'agent', 'is_active' => 1, 'code' => null];
  public $address = ['country' => 'SY', 'state' => null, 'city' => null, 'code' => null, 'phone' => null, 'email' => null, 'username' => null, 'password' => null];
  public $phone_code;
  public $title;
  public $selectedCountryName = 'SY';
  protected function messages()
  {
    return [
      'address.full_address.required_if' => __('alerts.address_required'),
    ];
  }
  protected function getListeners()
  {
    return [
      'deleteAgnent' => 'deleteAgnent',
      'editUser' => 'editUser',
      'deleteAlert' => 'deleteAlert',
      'DisableAgentToggle' => 'DisableAgentToggle',
      'openAddAgentModal' => 'openAddAgentModal',
      'restoreAlert' => 'restoreAlert',
      'restoreAgent' => 'restoreAgent',
    ];
  }
  public function restoreAlert($id)
  {
    $this->dispatch('areyousure', ['msg' => __("alerts.areyousure"), 'emitName' => 'restoreAgent', 'action' => __("actions.restore"), 'title' => __("alerts.restore"), 'data' => ['id' => $id]]);
  }
  public function restoreAgent($data)
  {
    $agent = Agent::withTrashed()->find($data['id']);
    $agent->restore();
    $agent->ledger_accounts()->restore();
    $agent->children()->restore();
    $this->dispatch("sweetalert:success", ['msg' => __("alerts.restored_successfully"), "title" => __("alerts.success"), "type" => "success"]);
    $this->dispatch("RefreshYajraDatatable", ['table' => 'agent-table']);
  }
  public function openAddAgentModal($action = "add", $agent_type = "agent")
  {
    $this->title = $action == "add" ? __("agents.add_new_agent") : __("agents.edit_agent");
    $types = ['agent', 'account'];
    if ($agent_type == "cashier") {
      $this->title = __("agents.add_new_cashier");
      $types = ['cashier'];
    }
    $this->types = $types;

    $this->agent = [
      'name' => '',
      'type' => $agent_type,
      'is_active' => 1,
      'code' => $this->generateAgentCode(),
    ];
    $this->address = [
      'country' => 'SY',
      'state' => null,
      'city' => null,
      'neighborhood' => null,
      'district' => null,
      'full_address' => '',
    ];
    $this->dispatch("open_modal", ['action' => "show", "id" => "AddNewAgentModal"]);
  }
  public function DisableAgentToggle($value, $checked)
  {
    $agent = Agent::find($value);
    $agent->is_active = $checked;
    $agent->save();
    $agent->users()->update(['status' => $checked]);
    event(new AgentForceLogout(1));
  }
  public function editUser($id)
  {
    $agent = Agent::with('address')->find($id)->toArray();
    $agent['password'] = '';
    $this->agent = $agent;

    $this->address = $agent['address'] ?? [];
    $this->dispatch("open_modal", ['action' => "show", "id" => "AddNewAgentModal"]);
  }
  public function deleteAlert($id)
  {
    $this->dispatch('areyousure', ['msg' => __("alerts.areyousure"), 'emitName' => 'deleteAgnent', 'action' => __("actions.delete"), 'title' => __("alerts.delete"), 'data' => ['id' => $id]]);
  }

  public function CopyCredentials()
  {
    $this->dispatch('copyToClipboard', [
      'text' => $this->agent['username'] . " " . $this->agent['password']
    ]);
  }
  public function deleteAgnent($data)
  {
    try {
      $id = $data['id'];
      $agent = Agent::findOrFail($id);

      // Check balance before deletion
      if ($agent->hasNonZeroBalance()) {
        throw new AgentHasBalanceException(
          __('alerts.Cannot delete :name with balance', ['name' => $agent->name]),
          $agent->getFormattedTotalBalance()
        );
      }

      // Use database transaction for data integrity
      DB::transaction(function () use ($agent) {
        // Delete related records
        $agent->ledger_accounts()->delete();
        $agent->children()->delete();

        // Mark as deleted and soft delete
        $agent->update(['code' => $agent->code . "_DELETED"]);
        $agent->delete();
      });

      // Success response
      $this->dispatch('sweetalert:success', [
        'msg' => __("alerts.deleted_successfully"),
        "title" => __("alerts.success"),
        "type" => "success"
      ]);

      $this->dispatch("RefreshYajraDatatable", ['table' => 'agent-table']);
    } catch (AgentHasBalanceException $e) {
      $this->dispatch('sweetalert:error', [
        'title' => __('alerts.Deletion Blocked'),
        'msg' => $e->getMessage(),
        'icon' => 'error'
      ]);
    } catch (\Exception $e) {
      Log::error('Agent deletion failed: ' . $e->getMessage(), [
        'agent_id' => $id ?? null,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
      ]);

      $this->dispatch("sweetalert:error", [
        'msg' => $e->getMessage(),
        "title" => "Error",
        "type" => "error"
      ]);
    } finally {
      // Always unblock UI regardless of success/failure
      $this->dispatch("UnBlockUI");
    }
  }

  public function  mount()
  {
    $this->agents = Agent::isAgent()->get();
    $this->phone_code = config("TransferSystem.default_phone_code") ?? "+963";
    $this->countries = CountriesFacade::lookup(app()->getLocale())->all();
    $this->agent['code'] = $this->generateAgentCode();
  }
  public function updated($name, $value)
  {

    if ($name == "address.country") {

      $this->phone_code = config("country_phone_codes.{$value}");
      $this->selectedCountryName = $value ?? '';
    }
  }
  public  function generateAgentCode()
  {
    $prefix = "AGT_";
    $type = $this->agent['type'];
    $prefix = ""; // $type == "agent" ? "AGT_" : ($type == "account" ? "ACC_" : "CSH_");

    // Get the last agent code and extract the number
    $lastAgent = Agent::orderBy('id', 'desc')->first();

    if ($lastAgent && $lastAgent->code) {
      // Extract number from last code (e.g., "AGT_100001" -> 100001)
      $lastNumber = (int) str_replace($prefix, '', $lastAgent->code);
      $nextNumber = $lastNumber + 1;
    } else {
      // If no agents exist, start with 100001
      $nextNumber = 100001;
    }

    return $prefix . $nextNumber;
  }
  public function addAgent()
  {


    $rules = [
      'agent.name' => 'required',
      'agent.phone' => 'nullable',

      'agent.email' => 'nullable|email|unique:agents,email,' . ($this->agent['id'] ?? null),
      'agent.code' => 'nullable|unique:agents,code,' . ($this->agent['id'] ?? null),
      'agent.username' => 'nullable|unique:users,username,' . ($this->agent['id'] ?? null),
      'address.country' => 'required',
      'address.city' => 'required',
      'address.neighborhood' => 'nullable',
      'address.district' => 'nullable',
      'address.full_address' => 'required_if:agent.type,agent',
    ];
    if ($this->agent['type'] == "agent") {
      $rules['agent.username'] = 'required|unique:agents,username,' . ($this->agent['id'] ?? null);
    }
    // Only require password if creating a new agent
    if (empty($this->agent['id']) && $this->agent['type'] == "agent") {
      $rules['agent.password'] = 'required|min:8';
    } else {
      // Password is optional when editing
      $rules['agent.password'] = 'nullable|min:8';
    }
    if ($this->agent['type'] == "cashier") {
      $rules['agent.username'] = 'required|unique:agents,username,' . ($this->agent['id'] ?? 'null');
      $rules['agent.parent_id'] = 'required';
    }

    try {
      // Validate the data
      $this->validate($rules);

      if (isset($this->agent['id'])) {
        $agent = Agent::find($this->agent['id']);
        $updatedData = $this->agent;

        if (!empty($updatedData['password'])) {
          $updatedData['password'] = bcrypt($updatedData['password']);
        } else {
          unset($updatedData['password']);
        }

        $agent->update($updatedData);
      } else {
        if ($this->agent['type'] == "agent")
          $this->agent['password'] = bcrypt($this->agent['password']);
        $agent = Agent::create($this->agent);
        $credit_limit_data = [
          'agent_id' => $agent->id,
          'currency_code' => 'USD',
          'send_transfer' => false,


        ];
        $agent->GlobalCreditLimit()->create($credit_limit_data);

        if ($agent->type != "account") {
          $user = new User();
          $user->name = $this->agent['name'];
          $user->phone = $this->agent['phone'] ?? "";
          $user->email = $this->agent['email'] ?? "";
          $user->username = $this->agent['username'] ?? null;
          $user->password = $this->agent['password'];
          $user->agent_id = $agent->id;
          $user->save();
        }
      }

      $agent->createOrUpdateAddress($this->address);

      $this->dispatch("RefreshYajraDatatable", ['table' => 'agent-table']);
      $this->dispatch("sweetalert:success", [
        'msg' => __("alerts.saved_successfully"),
        'title' => __("alerts.success"),
        'type' => "success"
      ]);
      $this->dispatch("open_modal", ['action' => "hide", "id" => "AddNewAgentModal"]);
    } catch (ValidationException $e) {
      $firstField = array_key_first($e->validator->errors()->messages());
      $firstErrorMessage = $e->validator->errors()->first($firstField);
      $this->dispatch('focus-error', ['field' => $firstField]);
      $this->dispatch("sweetalert:error", [
        'msg' => $firstErrorMessage,
        'title' => __("alerts.error"),
        'type' => "error"
      ]);
      throw $e; // rethrow so Livewire still shows errors
    } catch (\Exception $e) {
      $this->dispatch("sweetalert:error", [
        'msg' => $e->getMessage(),
        'title' => __("alerts.Error"),
        'type' => "error"
      ]);
    }
  }


  protected function failedValidation(\Illuminate\Validation\Validator $validator)
  {
    $firstField = array_key_first($validator->errors()->messages());
    $this->dispatchBrowserEvent('focus-error', ['field' => $firstField]);
  }
  public function render()
  {
    return view('livewire.agents.add');
  }
}
