<?php

namespace App\Livewire\Booking;

use Abivia\Ledger\Models\LedgerCurrency;
use App\Events\RefreshDataTable;
use App\Exceptions\InsufficientCurrencyBalance;
use App\Exceptions\InvalidExchangeRateException;
use App\Models\Agent;
use App\Models\Client;
use App\Models\Currency;
use App\Models\CustomLedgerAccount;
use App\Models\FormFieldConfig;
use App\Models\LedgerBooking;
use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\Ledger\LedgerService;
use App\Services\Transfer\BookingService;
use App\Services\Transfer\TransferService;
use App\Rules\NotChildOf;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class Add extends Component
{
  use WithFileUploads;

  // Form filters and type
  public array $filters = ['type' => 'booking'];
  public string $type = 'booking';
  public ?int $booking_id = null;
  public $needsConfirmation = false; // Track if confirmation is needed for losing transfer
  public $pendingTransferData = null; // Store transfer data while waiting for confirmation
  // Calculated values
  public float $totalToReceive = 0;
  public string $send_fee_type = 'fixed';
  public string $receiver_fee_type = 'fixed';
  public float $debit = 0;
  public float $credit = 0;
  public float $transferFee = 0;
  public float $total = 0;
  public float $revenue = 0;
  public bool $suppressCalculate = false;

  // Search configuration
  public string $searchUrl = '/api/agents/search';

  // Main transfer data
  #[Validate([
    'transfer.sender_id' => 'required|exists:agents,id',
    'transfer.receiver_id' => [
      'required',
      'exists:agents,id',
      'different:transfer.sender_id',
      new NotChildOf('transfer.sender_id')
    ],
    'transfer.currency' => 'required',
    'transfer.amount' => 'required|numeric',
    'transfer.send_fee' => 'required|numeric',
    'transfer.receiver_fee' => 'required|numeric',
    'transfer.date' => 'nullable|date',
    'transfer.recipient_name' => 'required|string',
  ])]
  public array $transfer = [
    'sender_id' => null,
    'receiver_id' => null,
    'currency' => 'USD',
    'delivery_currency' => 'USD',
    'delivery_amount' => 0,
    'amount' => null,
    'send_fee' => 0,
    'receiver_fee' => 0,
    'transfer_date' => null,
    'note' => null,
    'type' => 'booking',
    'exchange_rate' => 1,
    'send_fee_value' => 0,
    'receiver_fee_value' => 0,
    'recipient_name' => '',
  ];

  // Recipient information


  // Auto-filled sender details
  public array $senderDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
    'balance' => null,
  ];

  // Auto-filled receiver details
  public array $receiverDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
  ];

  // Available data
  public   $currencies = [];
  public array $senders = [];
  public array $receivers = [];
  public array $formFieldConfigs = [];
  public string $factor = 'multiply';

  // File attachments
  public array $attachments = [];

  protected function getListeners(): array
  {
    return [
      'editTransfer' => 'editBooking',
      'proceedWithLosingTransfer' => 'proceedWithLosingTransfer',
      'cancelLosingTransfer' => 'cancelLosingTransfer',
    ];
  }

  public function mount(): void
  {
    $this->initializeComponent();
  }

  private function initializeComponent(): void
  {
    $this->type = $this->filters['type'] ?? 'booking';
    $currencies = Currency::get();
    $this->currencies = $currencies;

    $this->transfer['date'] = now()->format('Y-m-d');
    $this->formFieldConfigs = FormFieldConfig::forForm('booking_form')->toArray();

    if ($this->booking_id) {
      $this->editBooking($this->booking_id);
    }
  }

  public function editBooking(int $booking_id): void
  {
    $transfer = LedgerBooking::findOrFail($booking_id);

    // Suppress calculations during editing
    $this->suppressCalculate = true;

    $this->populateSenderDetails($transfer);
    $this->populateTransferData($transfer);

    // Re-enable calculations after editing is complete
    $this->suppressCalculate = false;

    $this->dispatch('scrollToForm', ['form_id' => 'booking_form']);
  }

  private function populateSenderDetails(LedgerBooking $transfer): void
  {
    $this->senderDetails = [
      'name' => $transfer->sender?->name,
      'phone' => $transfer->sender?->phone,
      'address' => $transfer->sender?->address?->full_address,
      'balance' => 0,
    ];
  }

  private function populateReceiverDetails(LedgerBooking $transfer): void
  {
    $this->receiverDetails = [
      'name' => $transfer->receiver?->name,
      'phone' => $transfer->receiver?->phone,
      'address' => $transfer->receiver?->address?->full_address,
    ];
  }



  private function populateTransferData(LedgerBooking $transfer): void
  {
    // Suppress ALL calculations and property watchers during data loading
    $this->suppressCalculate = true;

    $this->transfer = $transfer->toArray();
    $this->transfer['status'] = $transfer->status;

    $this->transfer['receiver_fee_value'] = $transfer->receiver_fee;
    $this->transfer['send_fee_value'] = $transfer->send_fee;
    $this->send_fee_type = 'fixed';
    $this->receiver_fee_type = 'fixed';
    $this->transfer['exchange_rate'] = $transfer->rate;

    // Re-enable calculations after all data is loaded
    $this->suppressCalculate = false;
  }

  public function uploadAttachments(int $transfer_id): void
  {
    $transfer = LedgerBooking::findOrFail($transfer_id);

    if (empty($this->attachments)) {
      return;
    }

    $transfer->clearMediaCollection('attachments');

    foreach ($this->attachments as $attachment) {
      $transfer
        ->addMedia($attachment->getRealPath())
        ->usingFileName($attachment->getClientOriginalName())
        ->toMediaCollection('attachments');
    }
  }

  // Form field configuration methods
  public function fieldIsVisible(string $key): bool
  {
    return $this->formFieldConfigs[$key]['visible'] ?? true;
  }

  public function fieldIsRequired(string $key): bool
  {
    return $this->formFieldConfigs[$key]['required'] ?? false;
  }

  public function fieldIsLocked(string $key): bool
  {
    return $this->formFieldConfigs[$key]['always_required'] ?? false;
  }

  public function getFieldLabel(string $key): string
  {
    return $this->formFieldConfigs[$key]['label'] ?? Str::title($key);
  }

  // Updated methods
  public function updated(string $key, mixed $value): void
  {
    // Prevent calculations when suppressCalculate is true
    if ($this->suppressCalculate) {
      return;
    }

    // Only trigger calculations for specific fields, not all
    if ($key === "transfer.currency" || $key === "transfer.delivery_currency") {
      $this->GetRateandFactor();
    }

    // Only calculate for amount-related changes, not all property changes
    if (in_array($key, ["transfer.amount", "transfer.send_fee_value", "transfer.receiver_fee_value"])) {
      $this->Calculate();
    }
  }

  public function updatedTransferSenderId(mixed $value): void
  {
    if ($value) {
      $this->loadSenderDetails($value);
    } else {
      $this->resetSenderDetails();
    }
  }

  public function updatedTransferReceiverId(mixed $value): void
  {
    Log::info('updatedTransferReceiverId called with: ' . $value);
    Log::info('Current receiver_id: ' . ($this->transfer['receiver_id'] ?? 'null'));

    if ($value) {
      $this->loadReceiverDetails($value);
      $this->calculateFees();

      Log::info('After calculateFees, receiver_id: ' . ($this->transfer['receiver_id'] ?? 'null'));
    } else {
      $this->resetReceiverDetails();
    }
  }

  public function updatedTransferCurrency(): void
  {
    $this->transfer['delivery_currency'] = $this->transfer['currency'];
    $this->calculateFees();
    $this->getRateAndFactor();
    $this->calculate();
  }
  public function updatedTransferDeliveryCurrency(): void
  {

    $this->calculateFees();
    $this->getRateAndFactor();
    $this->calculate();
  }


  public function updatedTransferExchangeRate(mixed $value): void
  {
    $this->calculate();
  }

  public function updatedTransferDeliveryAmount(mixed $value): void
  {
    if ($this->suppressCalculate) {
      return;
    }

    $this->getRateAndFactor();
    $this->calculateDeliveryAmount($value);
  }

  public function updatedFactor(string $value): void
  {
    $this->calculate();
  }

  public function updatedSendFeeType(string $value): void
  {
    $this->send_fee_type = $value;
    $this->calculateFees();
  }

  public function updatedTransferSendFeeValue(string $value): void
  {
    $this->calculateFees();
  }

  public function updatedTransferReceiverFeeValue(string $value): void
  {
    $this->calculateFees();
  }

  public function updatedReceiveFeeType(string $value): void
  {
    $this->receiver_fee_type = $value;
    $this->calculateFees();
  }
  public function UpdatedTransferNotes(string $value): void
  {
    $this->transfer['receiver_note'] = $value;
  }

  // Private helper methods
  private function loadSenderDetails(int $senderId): void
  {
    $sender = Agent::with('address')->findOrFail($senderId);
    $this->senderDetails = [
      'name' => $sender->name,
      'phone' => $sender->phone,
      'address' => $sender->address?->full_address,
      'balance' => 0,
    ];
  }

  private function loadReceiverDetails(int $receiverId): void
  {
    $receiver = Agent::with('address')->findOrFail($receiverId);
    $this->receiverDetails = [
      'name' => $receiver->name,
      'phone' => $receiver->phone,
      'address' => $receiver->address?->full_address,
    ];
  }

  private function resetSenderDetails(): void
  {
    $this->senderDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
      'balance' => null,
    ];
  }

  private function resetReceiverDetails(): void
  {
    $this->receiverDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
    ];
  }

  private function calculateDeliveryAmount(mixed $value): void
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }
      $deliveryAmount = $value ?? 0;
      $fromCurrency = $this->transfer['currency'];
      $toCurrency = $this->transfer['delivery_currency'];

      if (!$deliveryAmount || !$toCurrency) {
        return;
      }

      $factor = $this->factor ?? 'multiply';
      $rate = $this->transfer['exchange_rate'] ?? 1;

      $this->suppressCalculate = true;

      if ($factor === 'divide') {
        $this->transfer['amount'] = $deliveryAmount * $rate;
      } else {
        $this->transfer['amount'] = $deliveryAmount / $rate;
      }

      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
    }
  }

  // Calculation methods
  public function calculate(): void
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }
      $feeType = $this->transfer['fee_type'] ?? 'collected';

      if ($feeType === 'included') {
        $this->transfer['amount'] = $this->transfer['amount'] - $this->transfer['send_fee'];
      }

      $fromAmount = $this->transfer['amount'] ?? 0;
      $fromCurrency = $this->transfer['currency'];
      $toCurrency = $this->transfer['delivery_currency'];

      if (!$fromAmount || !$fromCurrency) {
        return;
      }

      $factor = $this->factor;
      $rate = $this->transfer['exchange_rate'];

      $converter = new CurrencyConverter();
      $converted = $converter->convert(
        $fromAmount,
        $fromCurrency,
        $toCurrency,
        $this->currencies,
        'balance',
        $rate,
        $factor === 'multiply' ? '*' : '/'
      );
      $this->suppressCalculate = true;
      $this->transfer['delivery_amount'] = $converted;
      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
    } catch (\Exception $e) {
      Log::info($e);
      Log::error('Error in calculate method', ['error' => $e->getMessage()]);
      $this->addError('transfer.rate', 'An error occurred during calculation');
    }
  }

  public function getRateAndFactor(): void
  {
    try {
      $fromAmount = $this->transfer['amount'] ?? 0;
      $fromCurrency = $this->transfer['currency'];
      $toCurrency = $this->transfer['delivery_currency'];

      if (!$toCurrency || !$fromCurrency) {
        $this->setDefaultRateAndFactor();
        return;
      }

      if ($fromCurrency === $toCurrency) {
        $this->setDefaultRateAndFactor();
        return;
      }

      $factor = getCurrencyFactor($fromCurrency, $toCurrency, 'balance');
      $this->factor = $factor;
      $this->transfer['factor'] = $factor;

      $rate = exchange_rate($fromCurrency, $toCurrency, 'balance');
      $this->transfer['exchange_rate'] = $rate;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
    } catch (\Exception $e) {
      Log::error('Error in getRateAndFactor method', ['error' => $e->getMessage()]);
      $this->addError('transfer.rate', 'An error occurred while getting exchange rate');
    }
  }

  private function setDefaultRateAndFactor(): void
  {
    $this->factor = 'multiply';
    $this->transfer['factor'] = 'multiply';
    $this->transfer['exchange_rate'] = 1;
  }

  public function calculateFees(): void
  {
    $this->transfer['send_fee'] = $this->calculateFee(
      $this->send_fee_type,
      $this->transfer['send_fee_value'],
      $this->transfer['amount']
    );
    Log::info($this->transfer['send_fee']);

    $this->transfer['receiver_fee'] = $this->calculateFee(
      $this->receiver_fee_type,
      $this->transfer['receiver_fee_value'],
      $this->transfer['delivery_amount']
    );
  }

  private function calculateFee(string $feeType, float $feeValue, ?float $amount): float
  {

    return match ($feeType) {
      'fixed' => $feeValue,
      'percent_hundred' => ($amount ?? 0) * $feeValue / 100,
      'percent_thousand' => ($amount ?? 0) * $feeValue / 1000,
      'percent_ten_thousand' => ($amount ?? 0) * $feeValue / 10000,
      default => 0,
    };
  }

  public function calculateDebitAndCredit(): void
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    $senderFee = $this->transfer['send_fee'] ?? 0;

    $this->debit = $amount + $senderFee;
    $this->credit = $amount + $receiverFee;
    $this->revenue = $senderFee - $receiverFee;
  }

  public function getTotalToReceive(): float
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    $this->totalToReceive = $amount + $receiverFee;
    return $this->totalToReceive;
  }

  public function getTotalToSendProperty(): float
  {
    $amount = $this->transfer['amount'] ?? 0;
    $senderFee = $this->transfer['send_fee'] ?? 0;
    return $amount + $senderFee;
  }

  public function getAmountInWordsProperty(): string
  {
    if ($this->transfer['amount'] && $this->transfer['currency']) {
      return toCurrencyWords(
        floatval($this->transfer['amount']),
        $this->transfer['currency'],
        app()->getLocale()
      );
    }
    return '';
  }

  public function getAmountInWords(float $amount, string $currency): string
  {
    if ($amount && $currency) {
      return toCurrencyWords(
        floatval($amount),
        $currency,
        app()->getLocale()
      );
    }
    return '';
  }

  public function RoundAmount(string $key): void
  {
    $this->transfer[$key] = round($this->transfer[$key] ?? 0);
  }

  public function getBalances(?int $agentId)
  {
    try {
      if (!$agentId) {
        return [];
      }

      $agent = Agent::find($agentId);
      return $agent ? $agent->getAgentBalances() : [];
    } catch (\Exception $e) {
      Log::error('Error in getBalances method', ['error' => $e->getMessage()]);

      return [];
    }
  }

  // Main actions
  public function confirmTransfer(): void
  {
    try {
      $this->validate();
      $receiver = Agent::find($this->transfer['receiver_id']);
      if ($receiver->isitChild($this->transfer['sender_id'])) {
        $this->dispatch("sweetalert:error", ['msg' => __("Cannot Transfer To Your Branch"), "title" => "Error", "type" => "error"]);
        return;
      }

      if (isset($this->transfer['id'])) {
        $this->updateBooking();
        return;
      }
      $transfer = $this->transfer;
      $send_fee = $transfer['send_fee'] ?? 0;
      $receiver_fee = $transfer['receiver_fee'] ?? 0;

      if ($receiver_fee > $send_fee) {
        // Show confirmation dialog for losing transfer
        $this->needsConfirmation = true;
        $this->pendingTransferData = $this->transfer;
        $this->dispatch("areyousure", [
          'title' => __("Warning: Losing Transfer"),
          'msg' => __("Receiver fee is greater than send fee. This booking will be a LOSER. Are you sure you want to continue?"),
          'emitName' => 'proceedWithLosingTransfer',
          'action' => __("Continue with losing transfer")
        ]);
        return;
      }

      $this->createBooking();
    } catch (ValidationException $e) {
      $this->handleValidationError($e);
      throw $e;
    } catch (\Throwable $e) {
      Log::info($e);
      $this->dispatch('sweetalert:error', [
        'msg' => $e->getMessage(),
        'title' => __('Error'),
        'type' => 'error'
      ]);
    } catch (\Exception $e) {
      Log::error('Transfer confirmation failed', ['error' => $e->getMessage()]);
      $this->dispatch('sweetalert:error', [
        'msg' => $e->getMessage(),
        'title' => __('Error'),
        'type' => 'error'
      ]);
    }
  }
  public function proceedWithLosingTransfer()
  {
    if (!$this->needsConfirmation || !$this->pendingTransferData) {
      $this->dispatch("sweetalert:error", ['msg' => __("No pending transfer to confirm"), "title" => "Error", "type" => "error"]);
      return;
    }

    $this->needsConfirmation = false;
    $this->transfer = $this->pendingTransferData;
    $this->pendingTransferData = null;

    $this->createBooking();
  }

  public function cancelLosingTransfer()
  {
    $this->needsConfirmation = false;
    $this->pendingTransferData = null;
    $this->dispatch("sweetalert:info", ['msg' => __("Transfer cancelled"), "title" => "Cancelled", "type" => "info"]);
  }

  private function createBooking(): void
  {
    try {
      DB::beginTransaction();
      $transfer_array = $this->transfer;


      LedgerBooking::disableAuditing();
      $transfer_array['reference'] = generateReferenceNumber("REC");
      $transfer_array['secret'] = rand(1000, 9999);
      $transfer_array['user_id'] = Auth::id() ?? 1;
      $transfer = LedgerBooking::create($transfer_array);
      $service = new BookingService();
      $service->createTransferLedgerEntries($transfer);

      $this->dispatch('saveMedia', \App\Models\LedgerBooking::class, $transfer->id);
      $this->refreshDataTables();
      $this->showSuccessMessage($transfer);
      $this->resetForm();
      DB::commit();
    } catch (InsufficientCurrencyBalance $e) {
      DB::rollBack();
      $this->dispatch('sweetalert:error', [
        'msg' => $e->getMessage(),
        'title' => __('Error'),
        'type' => 'error'
      ]);
    } catch (\Exception $e) {
      Log::error('Transfer creation failed', ['error' => $e->getMessage()]);
      $this->dispatch('sweetalert:error', [
        'msg' => $e->getMessage(),
        'title' => __('Error'),
        'type' => 'error'
      ]);
      DB::rollBack();
    }
  }

  private function handleValidationError(ValidationException $e): void
  {
    $firstField = array_key_first($e->validator->errors()->messages());
    $firstErrorMessage = $e->validator->errors()->first($firstField);

    $this->dispatch('focus-error', ['field' => $firstField]);
    $this->dispatch('sweetalert:error', [
      'msg' => $firstErrorMessage,
      'title' => __('alerts.error'),
      'type' => 'error'
    ]);
  }

  private function refreshDataTables(): void
  {
    event(new RefreshDataTable('booking-table'));
    event(new RefreshDataTable('livewatch-table'));
    $this->dispatch('RefreshYajraDatatable', ['table' => 'booking-table']);
  }

  private function showSuccessMessage(LedgerBooking $transfer): void
  {
    $this->dispatch('toastr:success', [
      'msg' => __('transfers.transfer_created_successfully_toast', [
        'reference' => $transfer->reference,
        'amount' => $transfer->amount,
        'currency' => $transfer->currency
      ]),
      'reference' => $transfer->reference,
      'amount' => $transfer->amount,
      'currency' => $transfer->currency,
      'title' => __('alerts.success'),
      'type' => 'success',
      'position' => 'bottom-center'
    ]);
  }

  public function updateBooking(): void
  {
    try {

      $transfer = LedgerBooking::findOrFail($this->transfer['id']);

      LedgerBooking::disableAuditing();

      $transfer->update($this->transfer);
      LedgerBooking::enableAuditing();
      $service = new BookingService();
      $result = $service->UpdateLedger($transfer);

      $this->uploadAttachments($transfer->id);

      if ($result['success']) {
        $this->dispatch('sweetalert:success', [
          'msg' => $result['message'],
          'title' => __('Success'),
          'type' => 'success'
        ]);
        sleep(1);
        redirect()->to('/Transfer/Booking');
      } else {
        Log::error('Booking update failed', $result);
        $this->dispatch('sweetalert:error', [
          'msg' => $result['message'],
          'title' => __('Error'),
          'type' => 'error'
        ]);
      }
    } catch (\Exception $e) {
      Log::error('Error updating transfer', ['error' => $e->getMessage()]);
    }
  }

  public function cancel(): RedirectResponse
  {

    return redirect()->to('/Transfer/Booking');
  }

  public function resetForm(): void
  {
    $this->reset('transfer');
    $this->resetSenderDetails();
    $this->resetReceiverDetails();

    $this->transfer['currency'] = 'USD';
    $this->transfer['date'] = now()->format('Y-m-d');


    $this->resetErrorBag();
  }


  public function render()
  {
    return view('livewire.booking.add');
  }
}
