<?php

namespace App\Livewire\Components;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\LedgerTransfer;
use App\Models\User;
use App\Models\CustomLedgerAccount;
use App\Models\Currency;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class FloatingActionButton extends Component
{
  public $isOpen = false;
  public $actions = [];
  public $quickStats = [];
  public $currencyPositions = [];
  public $settings = [];
  public $recentActivity = [];
  public $config = [];
  public $isLoading = false;

  protected $listeners = [
    'refreshStats' => 'loadQuickStats',
    'toggleOffcanvas' => 'toggleOffcanvas',
    'updateActions' => 'loadActions'
  ];

  public function mount($config = [])
  {
    // Debug logging
    Log::info('FloatingActionButton component mounted', ['config' => $config]);

    $this->config = array_merge([
      'showStats' => true,
      'showSettings' => true,
      'showActivity' => true,
      'position' => 'middle-left', // bottom-right, bottom-left, top-right, top-left, middle-left, middle-right, center
      'theme' => 'primary',
      'size' => 'lg', // sm, md, lg
      'autoRefresh' => false,
      'refreshInterval' => 30000, // 30 seconds
    ], $config);

    $this->loadActions();
    $this->loadQuickStats();
    // Don't load currency positions on mount - load only when offcanvas opens
    $this->loadSettings();
    // Don't load recent activity on mount - load only when offcanvas opens

    if ($this->config['autoRefresh']) {
      $this->startAutoRefresh();
    }

    // Debug logging
    Log::info('FloatingActionButton component initialized', [
      'actions_count' => count($this->actions),
      'stats_count' => count($this->quickStats),
      'settings_count' => count($this->settings)
    ]);
  }

  public function loadActions()
  {
    // Default actions - can be overridden by config
    $this->actions = $this->config['actions'] ?? [
      [
        'id' => 'new-transfer',
        'title' => 'New Transfer',
        'icon' => 'ti ti-plus',
        'color' => 'primary',
        'action' => 'createTransfer',
        'route' => 'transfer.booking',
        'permission' => 'transfer.create'
      ],

      [
        'id' => 'export-data',
        'title' => 'Export Data',
        'icon' => 'ti ti-download',
        'color' => 'success',
        'action' => 'exportData',
        'permission' => 'data.export'
      ],
      [
        'id' => 'refresh-stats',
        'title' => 'Refresh Stats',
        'icon' => 'ti ti-refresh',
        'color' => 'warning',
        'action' => 'refreshStats',
        'permission' => null
      ],
      [
        'id' => 'quick-settings',
        'title' => 'Quick Settings',
        'icon' => 'ti ti-settings',
        'color' => 'secondary',
        'action' => 'openSettings',
        'permission' => 'settings.view'
      ]
    ];

    // Filter actions based on permissions - simplified for now
    $this->actions = array_filter($this->actions, function ($action) {
      // For now, show all actions. You can implement permission logic here later
      return true;
    });
  }

  public function loadQuickStats()
  {
    try {
      // Get real statistics from database
      $totalTransfers = LedgerTransfer::count();
      $completedTransfers = LedgerTransfer::where('status', 'completed')->count();
      $pendingTransfers = LedgerTransfer::where('status', 'pending')->count();
      $activeUsers = User::where('last_login_at', '>=', now()->subDays(7))->count();

      $this->quickStats = [
        [
          'title' => 'Total Transfers',
          'value' => number_format($totalTransfers),
          'icon' => 'ti ti-transfer',
          'color' => 'primary',
          'change' => '+12%',
          'changeType' => 'positive'
        ],
        [
          'title' => 'Completed',
          'value' => number_format($completedTransfers),
          'icon' => 'ti ti-check',
          'color' => 'success',
          'change' => '+8%',
          'changeType' => 'positive'
        ],
        [
          'title' => 'Pending',
          'value' => number_format($pendingTransfers),
          'icon' => 'ti ti-clock',
          'color' => 'warning',
          'change' => '-5%',
          'changeType' => 'negative'
        ],
        [
          'title' => 'Active Users',
          'value' => number_format($activeUsers),
          'icon' => 'ti ti-users',
          'color' => 'info',
          'change' => '+15%',
          'changeType' => 'positive'
        ]
      ];
    } catch (\Exception $e) {
      // Fallback to static data if database query fails
      $this->quickStats = [
        [
          'title' => 'Total Transfers',
          'value' => '1,234',
          'icon' => 'ti ti-transfer',
          'color' => 'primary'
        ],
        [
          'title' => 'Completed',
          'value' => '1,100',
          'icon' => 'ti ti-check',
          'color' => 'success'
        ],
        [
          'title' => 'Pending',
          'value' => '134',
          'icon' => 'ti ti-clock',
          'color' => 'warning'
        ],
        [
          'title' => 'Active Users',
          'value' => '25',
          'icon' => 'ti ti-users',
          'color' => 'info'
        ]
      ];
    }
  }

  public function loadCurrencyPositions()
  {
    try {
      // Get currency positions from CustomLedgerAccount similar to CurrencyPositionsDataTable
      $currencyAccounts = CustomLedgerAccount::where("category", 0)
        ->whereNotNull("extra->currency_code")
        ->with("balances", "names")
        ->get();

      $this->currencyPositions = [];

      foreach ($currencyAccounts as $account) {
        $extra = is_string($account->extra) ? json_decode($account->extra, true) : $account->extra;
        $currencyCode = $extra['currency_code'] ?? null;

        if (!$currencyCode) continue;

        // Get currency name
        $currency = Currency::where('code', $currencyCode)->first();
        $currencyName = $currency ? $currency->getTranslation('name', app()->getLocale()) : $currencyCode;

        // Get currency amount
        $currencyBalance = $account->balances->firstWhere('currency', $currencyCode);
        $currencyAmount = $currencyBalance ? round(-1 * $currencyBalance->balance, 0) : 0;

        // Get base amount (USD)
        $baseBalance = $account->balances->firstWhere('currency', "USD");
        $baseAmount = $baseBalance ? round(-1 * $baseBalance->balance, 0) : 0;

        // Get cost (if function exists)
        $cost = 0;
        if (function_exists('getExchangeCostPerUnitForTable')) {
          $cost = getExchangeCostPerUnitForTable("USD", $currencyCode);
        }

        // Only add if there's a meaningful amount
        if ($currencyAmount != 0 || $baseAmount != 0) {
          $this->currencyPositions[] = [
            'currency_code' => $currencyCode,
            'currency_name' => $currencyName,
            'currency_amount' => $currencyAmount,
            'base_amount' => $baseAmount,
            'cost' => $cost,
            'color' => $currencyAmount >= 0 ? 'success' : 'danger',
            'icon' => 'ti ti-currency-' . strtolower($currencyCode)
          ];
        }
      }

      // Limit to top 4 currency positions for display
      $this->currencyPositions = array_slice($this->currencyPositions, 0, 4);
    } catch (\Exception $e) {
      Log::error('Error loading currency positions: ' . $e->getMessage());
      $this->currencyPositions = [];
    }
  }

  public function loadSettings()
  {
    $this->settings = [
      [
        'id' => 'notifications',
        'title' => 'Enable Notifications',
        'description' => 'Receive real-time notifications',
        'value' => session('notifications_enabled', true),
        'icon' => 'ti ti-bell'
      ],
      [
        'id' => 'auto-refresh',
        'title' => 'Auto Refresh',
        'description' => 'Automatically refresh data every 30 seconds',
        'value' => session('auto_refresh_enabled', false),
        'icon' => 'ti ti-refresh'
      ],
      [
        'id' => 'dark-mode',
        'title' => 'Dark Mode',
        'description' => 'Switch to dark theme',
        'value' => session('dark_mode_enabled', false),
        'icon' => 'ti ti-moon'
      ],
      [
        'id' => 'compact-view',
        'title' => 'Compact View',
        'description' => 'Use compact layout for better performance',
        'value' => session('compact_view_enabled', false),
        'icon' => 'ti ti-layout-compact'
      ]
    ];
  }

  public function loadRecentActivity()
  {
    try {
      // Get recent transfers
      $recentTransfers = LedgerTransfer::with('sender')
        ->latest()
        ->take(5)
        ->get();

      $this->recentActivity = $recentTransfers->map(function ($transfer) {
        return [
          'id' => $transfer->id,
          'type' => 'transfer',
          'title' => "Transfer #{$transfer->id} " . ucfirst($transfer->status),
          'description' => $transfer->sender->name ?? 'Unknown User',
          'time' => $transfer->created_at->diffForHumans(),
          'icon' => $this->getStatusIcon($transfer->status),
          'color' => $this->getStatusColor($transfer->status)
        ];
      })->toArray();
    } catch (\Exception $e) {
      // Fallback to static data
      $this->recentActivity = [
        [
          'type' => 'transfer',
          'title' => 'Transfer #1234 completed',
          'description' => 'John Doe',
          'time' => '2 minutes ago',
          'icon' => 'ti ti-check',
          'color' => 'success'
        ],
        [
          'type' => 'transfer',
          'title' => 'New transfer request received',
          'description' => 'Jane Smith',
          'time' => '5 minutes ago',
          'icon' => 'ti ti-clock',
          'color' => 'warning'
        ],
        [
          'type' => 'user',
          'title' => 'New user registered',
          'description' => 'Mike Johnson',
          'time' => '10 minutes ago',
          'icon' => 'ti ti-user',
          'color' => 'info'
        ]
      ];
    }
  }

  private function getStatusIcon($status)
  {
    return match ($status) {
      'completed' => 'ti ti-check',
      'pending' => 'ti ti-clock',
      'cancelled' => 'ti ti-x',
      'failed' => 'ti ti-alert-triangle',
      default => 'ti ti-circle'
    };
  }

  private function getStatusColor($status)
  {
    return match ($status) {
      'completed' => 'success',
      'pending' => 'warning',
      'cancelled' => 'danger',
      'failed' => 'danger',
      default => 'secondary'
    };
  }

  public function toggleOffcanvas()
  {
    $this->isOpen = !$this->isOpen;

    if ($this->isOpen) {
      // Show loading state
      $this->isLoading = true;

      // Load heavy data only when offcanvas is opened
      $this->loadQuickStats();
      $this->loadCurrencyPositions();
      $this->loadRecentActivity();

      // Hide loading state
      $this->isLoading = false;
    } else {
      // Clear heavy data when offcanvas is closed to free memory
      $this->currencyPositions = [];
      $this->recentActivity = [];
      $this->isLoading = false;
    }
  }

  public function createTransfer()
  {
    if (isset($this->actions[0]['route'])) {
      return redirect()->route($this->actions[0]['route']);
    }
    return redirect()->route('transfer.booking');
  }

  public function exportData()
  {
    // Implement export functionality
    $this->dispatch('showNotification', [
      'type' => 'success',
      'message' => 'Data export started successfully! Check your downloads folder.'
    ]);
  }

  public function refreshStats()
  {
    $this->loadQuickStats();
    $this->loadCurrencyPositions();
    $this->loadRecentActivity();
    $this->dispatch('showNotification', [
      'type' => 'info',
      'message' => 'Statistics refreshed successfully!'
    ]);
  }

  public function openSettings()
  {
    $this->dispatch('showNotification', [
      'type' => 'info',
      'message' => 'Settings panel opened!'
    ]);
  }



  public function toggleSetting($settingId)
  {
    foreach ($this->settings as &$setting) {
      if ($setting['id'] === $settingId) {
        $setting['value'] = !$setting['value'];
        session(["{$settingId}_enabled" => $setting['value']]);
        break;
      }
    }

    // Apply setting changes
    $this->applySetting($settingId, $setting['value'] ?? false);
  }

  public function applySetting($settingId, $value)
  {
    switch ($settingId) {
      case 'notifications':
        // Apply notification setting
        $this->dispatch('toggleNotifications', $value);
        break;
      case 'auto-refresh':
        // Apply auto refresh setting
        if ($value) {
          $this->startAutoRefresh();
        } else {
          $this->stopAutoRefresh();
        }
        break;
      case 'dark-mode':
        // Apply dark mode setting
        $this->dispatch('toggleDarkMode', $value);
        break;
      case 'compact-view':
        // Apply compact view setting
        $this->dispatch('toggleCompactView', $value);
        break;
    }

    $this->dispatch('showNotification', [
      'type' => 'success',
      'message' => 'Setting updated successfully!'
    ]);
  }

  private function startAutoRefresh()
  {
    $this->dispatch('startAutoRefresh', [
      'interval' => $this->config['refreshInterval']
    ]);
  }

  private function stopAutoRefresh()
  {
    $this->dispatch('stopAutoRefresh');
  }

  public function render()
  {
    return view('livewire.components.floating-action-button');
  }
}
