<?php

namespace App\Livewire\Ledger;

use Livewire\Component;
use App\Models\CustomLedgerAccount;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class LedgerAccountForm extends Component
{
  public $mode = 'create'; // or edit
  public $ledgerUuid;
  public $parentUuid;
  public $agentId;
  public $code;
  public $name;
  public $debit = false;
  public $credit = false;
  public $category = false;

  protected $listeners = ['open_modal'];

  // Radio button handlers for credit/debit (mutually exclusive)
  public function updatedDebit($value)
  {
    if ($value) {
      $this->credit = false;
      // Don't affect category - it's independent
    }
  }

  public function updatedCredit($value)
  {
    if ($value) {
      $this->debit = false;
      // Don't affect category - it's independent
    }
  }

  // Category is a checkbox, so no special handling needed
  // It can be selected independently of credit/debit

  public function open_modal($data)
  {
    if ($data['id'] === 'ledgerAccountModal' && $data['action'] === 'show') {
      $this->openAccountModal($data['data'] ?? []);
    }
  }

  public function openAccountModal($data)
  {
    $this->resetErrorBag();
    $this->reset(['mode', 'ledgerUuid', 'parentUuid', 'agentId', 'code', 'name', 'debit', 'credit', 'category']);

    $this->mode = $data['mode'] ?? 'create';
    $this->agentId = $data['agentId'] ?? null;
    $this->parentUuid = $data['parentUuid'] ?? null;

    if ($this->mode === 'edit' && isset($data['ledgerUuid'])) {
      $account = CustomLedgerAccount::with('names')->where('ledgerUuid', $data['ledgerUuid'])->firstOrFail();
      $this->ledgerUuid = $account->ledgerUuid;
      $this->code = $account->code;
      $this->name = collect($account->names)->first()?->name ?? '';
      $this->debit = $account->debit;
      $this->credit = $account->credit;
      $this->category = $account->category;
    }
  }

  public function save()
  {
    // Debug: Log the form data
    Log::info('Form submission started', [
      'mode' => $this->mode,
      'code' => $this->code,
      'name' => $this->name,
      'debit' => $this->debit,
      'credit' => $this->credit,
      'category' => $this->category,
      'parentUuid' => $this->parentUuid,
      'agentId' => $this->agentId,
    ]);

    $this->validate([
      'code' => 'required|string|max:20',
      'name' => 'required|string|max:255',
      'debit' => 'boolean',
      'credit' => 'boolean',
      'category' => 'boolean',
    ]);

    // Validate that either credit or debit is selected (category is optional)
    if (!$this->debit && !$this->credit) {
      $this->addError('debit', 'يجب تحديد نوع الحساب (مدين أو دائن)');
      return;
    }

    try {
      if ($this->mode === 'edit') {
        $account = CustomLedgerAccount::where('ledgerUuid', $this->ledgerUuid)->firstOrFail();
        $account->code = $this->code;
        $account->debit = $this->debit;
        $account->credit = $this->credit;
        $account->category = $this->category;
        $account->save();

        $account->names()->updateOrCreate(
          ['language' => app()->getLocale()],
          ['name' => $this->name]
        );
      } else {
        $account = CustomLedgerAccount::create([
          'agent_id' => $this->agentId,
          'parentUuid' => $this->parentUuid,
          'code' => $this->code,
          'ledgerUuid' => (string) Str::uuid(),
          'debit' => $this->debit,
          'credit' => $this->credit,
          'category' => $this->category,
        ]);

        $account->names()->create([
          'language' => app()->getLocale(),
          'name' => $this->name,
        ]);
      }

      Log::info('Account saved successfully', ['account_id' => $account->id ?? $account->ledgerUuid]);

      $this->dispatch('notify', [
        'type' => 'success',
        'message' => $this->mode === 'edit' ? 'تم تحديث الحساب بنجاح.' : 'تم إنشاء الحساب بنجاح.'
      ]);

      $this->dispatch('refreshTree');
      $this->dispatch('open_modal', ['action' => 'hide', 'id' => 'ledgerAccountModal']);
    } catch (\Exception $e) {
      Log::error('Error saving ledger account: ' . $e->getMessage(), [
        'trace' => $e->getTraceAsString()
      ]);
      $this->dispatch('notify', [
        'type' => 'error',
        'message' => 'حدث خطأ أثناء حفظ الحساب: ' . $e->getMessage()
      ]);
    }
  }

  public function render()
  {
    return view('livewire.ledger.ledger-account-form');
  }
}
