<?php

namespace App\Livewire\Ledger;

use Livewire\Component;
use App\Models\Agent;
use Abivia\Ledger\Models\LedgerAccount;
use App\Models\CustomLedgerAccount;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class Tree extends Component
{
  public $agentId; // الفرع (agent) الذي نريد عرض حساباته
  public $search = '';
  public $expandedNodes = [];

  protected $listeners = [
    'refreshTree' => '$refresh',
  ];

  public function mount($agentId = null)
  {
    $this->agentId = $agentId;
  }

  // نحصل على الشجرة متداخلة فقط للحسابات التي تخص الفرع الحالي
  public function getAccountsProperty()
  {
    // نبدأ بالحسابات الرئيسية (parentUuid = null) للفرع
    return CustomLedgerAccount::with(['names'])
      ->whereNull('parentUuid')
      ->get();
  }

  // Filtered accounts based on search
  public function getFilteredAccountsProperty()
  {
    if (empty($this->search)) {
      return $this->accounts;
    }

    return $this->accounts->filter(function ($account) {
      return $this->matchesSearch($account, $this->search);
    });
  }

  // Recursive search function
  protected function matchesSearch($account, $search)
  {
    $search = strtolower($search);

    // Check account name
    $name = $account->names->firstWhere('language', app()->getLocale())?->name
      ?? $account->names->first()?->name
      ?? '';

    if (
      str_contains(strtolower($name), $search) ||
      str_contains(strtolower($account->code), $search)
    ) {
      return true;
    }

    // Check children recursively
    $children = CustomLedgerAccount::where('parentUuid', $account->ledgerUuid)->get();
    foreach ($children as $child) {
      if ($this->matchesSearch($child, $search)) {
        return true;
      }
    }

    return false;
  }

  // Statistics properties
  public function getTotalAccountsProperty()
  {
    return CustomLedgerAccount::count();
  }

  public function getRootAccountsProperty()
  {
    return CustomLedgerAccount::whereNull('parentUuid')->count();
  }

  public function getChildAccountsProperty()
  {
    return CustomLedgerAccount::whereNotNull('parentUuid')->count();
  }

  public function getMaxDepthProperty()
  {
    return $this->calculateMaxDepth($this->accounts);
  }

  protected function calculateMaxDepth($accounts, $currentDepth = 1)
  {
    $maxDepth = $currentDepth;

    foreach ($accounts as $account) {
      if ($account->children->isNotEmpty()) {
        $depth = $this->calculateMaxDepth($account->children, $currentDepth + 1);
        $maxDepth = max($maxDepth, $depth);
      }
    }

    return $maxDepth;
  }

  public function expandAll()
  {
    $this->expandedNodes = $this->getAllNodeIds($this->accounts);
    $this->dispatch('expandAll');
  }

  public function collapseAll()
  {
    $this->expandedNodes = [];
    $this->dispatch('collapseAll');
  }

  protected function getAllNodeIds($accounts)
  {
    $ids = [];

    foreach ($accounts as $account) {
      if ($account->children->isNotEmpty()) {
        $ids[] = $account->ledgerUuid;
        $ids = array_merge($ids, $this->getAllNodeIds($account->children));
      }
    }

    return $ids;
  }

  public function addChildAccount($parentUuid)
  {
    $this->dispatch('open_modal', [
      'action' => 'show',
      'id' => 'ledgerAccountModal',
      'data' => [
        'mode' => 'create',
        'parentUuid' => $parentUuid,
        'agentId' => $this->agentId,
      ]
    ]);
  }

  public function editAccount($ledgerUuid)
  {
    $this->dispatch('open_modal', [
      'action' => 'show',
      'id' => 'ledgerAccountModal',
      'data' => [
        'mode' => 'edit',
        'ledgerUuid' => $ledgerUuid,
      ]
    ]);
  }

  public function deleteAccount($ledgerUuid)
  {
    try {
      $account = CustomLedgerAccount::where('ledgerUuid', $ledgerUuid)->first();

      if (!$account) {
        $this->dispatch('notify', [
          'type' => 'error',
          'message' => 'الحساب غير موجود.'
        ]);
        return;
      }

      // Check if account has children
      $hasChildren = CustomLedgerAccount::where('parentUuid', $ledgerUuid)->exists();

      if ($hasChildren) {
        $this->dispatch('notify', [
          'type' => 'error',
          'message' => 'لا يمكن حذف الحساب لأنه يحتوي على حسابات فرعية. يرجى حذف الحسابات الفرعية أولاً.'
        ]);
        return;
      }

      // Check if account has transactions/balances
      $hasTransactions = $this->checkAccountHasTransactions($ledgerUuid);

      if ($hasTransactions) {
        $this->dispatch('notify', [
          'type' => 'error',
          'message' => 'لا يمكن حذف الحساب لأنه يحتوي على معاملات أو أرصدة.'
        ]);
        return;
      }

      // Delete the account
      $account->delete();

      $this->dispatch('notify', [
        'type' => 'success',
        'message' => 'تم حذف الحساب بنجاح.'
      ]);

      $this->dispatch('refreshTree');
    } catch (\Exception $e) {
      Log::error('Error deleting ledger account: ' . $e->getMessage());
      $this->dispatch('notify', [
        'type' => 'error',
        'message' => 'حدث خطأ أثناء حذف الحساب.'
      ]);
    }
  }

  /**
   * Check if account has any transactions or balances
   */
  protected function checkAccountHasTransactions($ledgerUuid)
  {
    // Check for ledger balances
    $hasBalances = DB::table('ledger_balances')
      ->where('ledgerUuid', $ledgerUuid)
      ->exists();

    if ($hasBalances) {
      return true;
    }

    // Check for journal details (transactions)
    $hasTransactions = DB::table('ledger_journal_details')
      ->where('ledgerUuid', $ledgerUuid)
      ->exists();

    return $hasTransactions;
  }

  /**
   * Add root account (no parent)
   */
  public function addRootAccount()
  {
    $this->dispatch('open_modal', [
      'action' => 'show',
      'id' => 'ledgerAccountModal',
      'data' => [
        'mode' => 'create',
        'parentUuid' => null,
        'agentId' => $this->agentId,
      ]
    ]);
  }

  public function render()
  {
    return view('livewire.ledger.tree');
  }
}
