<?php

namespace App\Livewire\Transfers;

use Abivia\Ledger\Models\LedgerCurrency;
use App\Events\RefreshDataTable;
use App\Exceptions\CreditLimitExceededException;
use App\Exceptions\InvalidExchangeRateException;
use App\Livewire\Components\Transfer;
use App\Models\Agent;
use App\Models\ApiConfig;
use App\Models\AttachmentType;
use App\Models\Client;
use App\Models\Currency;
use App\Models\LedgerTransfer;
use App\Services\Transfer\TransferService;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;
use App\Models\FormFieldConfig;
use App\Models\Reason;
use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\TransferCreditLimitValidator;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class Add extends Component
{
  use WithFileUploads;
  #[Validate('mimes:jpeg,png,jpg,pdf|max:1024')] // 1MB Max
  public $totalToReceive = 0;
  public $debit = 0, $credit = 0, $transferFee = 0, $total = 0, $revenue = 0;
  public $attachmentTypes = [];
  public $senderAgent, $receiverAgent;
  public $clientsSearchUrl = "/Client/Search";
  public $searchUrl = "/Client/Search";
  public  $source = "internal";
  public $send_flag = false;
  public  $agentId = null;
  public  $external_flag = 0;
  public $type = "transfer";
  public  $suppressCalculate = false;
  public  $reasons = [];
  public $transfer = [
    'sender_id' => null,
    'receiver_id' => null,
    'currency' => null,
    'delivery_currency' => null,
    'amount' => null,
    'sender_fee' => 0,
    'send_fee' => 0,
    'receiver_fee' => 0,
    'transfer_date' => null,
    'note' => null,
    'type' => 'transfer',
    'rate' => 1,
    'external_ref' => null,
    'delivery_amount' => null,
    "reason_id" => 1,
    "fee_type" => "collected"
  ];
  public  $recipient = [
    'name' => '',
    'phone' => '',
    'sender_identity_number' => '',
  ];
  // Auto-filled sender details
  public $senderDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
    'balance' => null,
  ];

  // Auto-filled receiver details
  public $receiverDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
  ];
  public $formFieldConfigs = [];
  // Available currencies and agents
  public $currencies = [];
  public $senders = [];
  public $receivers = [];
  public $address = null;
  public  $attatchments = [];
  public  $attachment_types = [];
  public  $add_client = false;
  public  $attachment_type = "attachments";

  protected $rules = [
    'transfer.sender_id' => 'required|exists:agents,id',
    'transfer.receiver_id' => 'required| different:transfer.sender_id',
    'transfer.currency' => 'required|exists:ledger_currencies,code',
    'transfer.amount' => 'required|numeric|min:1',
    //  'transfer.send_fee' => 'required|numeric|min:0',
    //    'transfer.delivery_amount' => 'required|numeric|min:1',
    'transfer.receiver_fee' => 'required|numeric|min:0',
    'transfer.transfer_date' => 'nullable|date',
    'recipient.name' => 'required',
    'recipient.phone' => 'required',
  ];


  protected function  getListeners()
  {
    return [
      'getClientData' => 'getClientData',
      'refreshAttachmentTypes' => 'refreshAttachmentTypes',
      'editTransfer' => 'editTransfer',
      'newTransfer' => 'newTransfer',
      "CancelAddClientEvent" => "CancelAddClientEvent",

    ];
  }

  public function  CancelAddClientEvent()
  {
    $this->add_client = false;
  }
  public  function  AddNewClientEvent()
  {

    $this->add_client = true;
  }
  public function newTransfer()
  {
    $this->recipient = [
      'name' => '',
      'phone' => '',
      'sender_identity_number' => '',
    ];
    $this->transfer = [
      'sender_id' => null,
      'receiver_id' => null,
      'currency' => 'USD', // default currency
      'delivery_currency' => 'USD',
      'amount' => 0,
      'sender_fee' => 0,
      'send_fee' => 0,
      'receiver_fee' => 0,
      'transfer_date' => null,
      'note' => null,
      'type' => 'transfer',
      'rate' => 1,
      'external_ref' => null,
      'reason_id' => 1
    ];
  }


  public function editTransfer($id)
  {

    $transfer = LedgerTransfer::where("id", $id)->first();
    $receiver = $transfer->receiver;
    $sender = $transfer->sender;
    $this->senderDetails = [
      'name' => $sender?->name,
      'phone' => $sender?->phone,
      'address' => $sender?->address?->full_address,
    ];
    $this->receiverDetails = [
      'name' => $receiver?->name,
      'phone' => $receiver?->phone,
      'address' => $receiver?->address?->full_address,
    ];
    $this->receiverAgent = $receiver;
    $this->transfer = $transfer->toArray();
    if ($transfer->isExternalTransfer()) {
      $this->source = "external";
      $this->external_flag = 1;
      $this->transfer['receiver_id'] = $this->transfer["api_id"] . "_" . $this->transfer["external_agent_id"];
    } else {
      $this->external_flag = 0;
      $this->source = "internal";
      $this->transfer['receiver_id'] = "internal_" . $this->transfer["receiver_id"];
    }
    $this->recipient = $transfer->customer_details;
    $this->recipient['sender_identity_number'] = $this->recipient['identity_number'] ?? null;
    $this->searchSender();
  }
  public function  refreshAttachmentTypes()
  {
    $this->attachmentTypes = AttachmentType::all();
  }
  public function  UploadAttatchments($transfer_id)
  {
    $transfer = LedgerTransfer::find($transfer_id);
    $attachment_type = $this->attachment_type;
    if (count($this->attatchments) > 0) {
      $transfer->clearMediaCollection($attachment_type ?? 'attatchments');
      foreach ($this->attatchments as $attatchment) {

        $transfer
          ->addMedia($attatchment->getRealPath())
          ->usingFileName($attatchment->getClientOriginalName())
          ->toMediaCollection($attachment_type ?? 'attatchments');
      }
    }
  }
  public function  searchSender()
  {
    $recipent = $this->recipient;

    $this->validate([
      'recipient.sender_identity_number' => 'required',
    ]);


    $client = Client::where('phone', '=', $recipent['sender_identity_number'])->orWhere('identity_number', '=', $recipent['sender_identity_number'])->orWhere('name', '=', $recipent['sender_identity_number'])->orWhere('id', '=', $recipent['sender_identity_number'])->first();
    if ($client) {
      $this->recipient = $client->toArray();
      $this->recipient['sender_client_id'] = $client->id;
      //  $this->recipient['sender_identity_number'] = $client->identity_number;
      $this->transfer['client_id'] = $client->id;
      $this->transfer['client_sender_id_number'] =  $client->identity_number;

      $this->dispatch("LoadClientData", ['id' => $client->id]);
    } else {
      $this->dispatch("sweetalert:error", ['msg' => "No Client Found", "title" => "Error", "type" => "error"]);
    }
  }
  public  function  getClientData($id)
  {
    $id = $id['id'];
    $client = Client::find($id);

    $this->recipient['recipient_name'] = $client->name;
    $this->recipient['recipient_phone'] = $client->phone;
    $this->recipient['sender_client_identity_number'] = $client->identity_number;
    $this->recipient['sender_client_id'] = $client->id ?? null;
    $this->recipient['sender_identity_number'] = $client->identity_number ?? null;
    $this->recipient['identity_number'] = $client->id;
    $this->add_client = false;
  }
  public function mount($transfer_id = null)
  {
    $this->currencies = Currency::get();

    $this->senders = LedgerTransfer::GetSourceAgents();
    $this->receivers = LedgerTransfer::GetDestinationAgents();
    $this->reasons = Reason::get();
    $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
    $this->attachmentTypes = AttachmentType::get();
    $this->formFieldConfigs = FormFieldConfig::forForm('transfer_form');
    if ($transfer_id) {
      $this->editTransfer($transfer_id);
    }
  }
  public function Updated($key, $value)
  {

    $this->validateOnly($key);
    if (in_array($key, ['transfer.currency', 'transfer.amount', 'transfer.sender_id'])) {
      $this->CheckBalance();
    }
    if ($key != "transfer.amount" && $key != "transfer.delivery_amount" && $key != "transfer.rate" && $key != "transfer.rate_factor") {
      $this->GetRateandFactor();
    }
    if ($key != "transfer.rate" && $key != "transfer.delivery_amount")
      $this->Calculate();
  }
  public function UpdatedTransferDeliveryAmount($value)
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }

      $converter = new CurrencyConverter();
      $delivery_amount = $value ?? 0;
      $from_currency = $this->transfer['currency'];
      $to_currency = $this->transfer['delivery_currency'];

      if (!$delivery_amount || !$to_currency) {
        return;
      }
      $factor = $this->transfer['rate_factor'];
      $rate = $this->transfer['rate'];
      $this->suppressCalculate = true;

      if ($factor === 'divide') {
        $this->transfer['amount'] = $delivery_amount * $rate;
      } else {
        $this->transfer['amount'] =  $delivery_amount / $rate;
      }

      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    }
  }

  public  function  GetRateandFactor()
  {
    try {
      $from_amount = $this->transfer['amount'] ?? 0;
      $from_currency = $this->transfer['currency'];
      $to_currency = $this->transfer['delivery_currency'];

      if (!$to_currency || !$from_currency) {

        $this->transfer['rate_factor'] = "multiply";
        $this->transfer['rate'] = 1;
        return;
      }
      if ($from_currency == $to_currency) {
        $this->transfer['rate_factor'] = "multiply";
        $this->transfer['rate'] = 1;
        return;
      }
      $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');
      $this->transfer['rate_factor'] = $factor;
      $rate = exchange_rate($from_currency, $to_currency,   'transfer');
      $this->transfer['rate'] = $rate;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    } catch (\Exception $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    }
  }

  public function Calculate()
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }

      $fee_type = $this->transfer['fee_type'] ?? "collected";
      if ($fee_type == "included") {
        $this->transfer['amount'] = $this->transfer['amount'] - $this->transfer['send_fee'];
      }
      $converter = new CurrencyConverter();
      $from_amount = $this->transfer['amount'] ?? 0;
      $from_currency = $this->transfer['currency'] ?? "EUR";
      $to_currency = $this->transfer['delivery_currency'] ?? "EUR";

      if (!$from_amount || !$from_currency) {

        return;
      }
      $factor = $this->transfer['rate_factor'];
      $rate = $this->transfer['rate'];

      $converted = $converter->convert(
        $from_amount,
        $from_currency,
        $to_currency,
        $this->currencies,
        'transfer',
        $rate,
        $factor
      );
      $this->suppressCalculate = true;


      $this->transfer['delivery_amount'] = $converted;
      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    } catch (\Exception $e) {
    }
  }
  public  function UpdatedTransferRate($value)
  {

    $converter = new CurrencyConverter();
    $from_amount = $this->transfer['amount'] ?? 0;
    $from_currency = $this->transfer['currency'] ?? "EUR";
    $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
    $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');

    $this->transfer['rate_factor'] = $factor;
    $converted = $converter->convert(
      $from_amount,
      $from_currency,
      $to_currency,
      $this->currencies,
      'transfer',
      $value,
      $factor == "multiply" ? "*" : "/"
    );

    $this->transfer['delivery_amount'] = $converted;
  }
  public  function UpdatedTransferRateFactor($value)
  {

    $converter = new CurrencyConverter();
    $from_amount = $this->transfer['amount'] ?? 0;
    $from_currency = $this->transfer['currency'] ?? "EUR";
    $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
    $factor = $value; // getCurrencyFactor($from_currency, $to_currency, 'transfer');
    $rate = $this->transfer['rate'];
    $this->transfer['rate_factor'] = $factor;
    $converted = $converter->convert(
      $from_amount,
      $from_currency,
      $to_currency,
      $this->currencies,
      'transfer',
      $rate,
      $factor == "multiply" ? "*" : "/"
    );

    $this->transfer['delivery_amount'] = $converted;
  }
  public function UpdatedTransferDeliveryCurrency($value)
  {
    $this->CheckCanSendReceiveAgent();
    // Update delivery currency when transfer currency changes
    $this->transfer['delivery_currency'] = $value;

    $this->GetRateandFactor();
  }

  public function UpdatedTransferAmount($value)
  {
    try {


      $converter = new CurrencyConverter();
      $amount = $value ?? 0;
      $from_currency = $this->transfer['currency'];
      $to_currency = $this->transfer['delivery_currency'];

      if (!$amount || !$from_currency) {
        return;
      }
      $factor = $this->transfer['rate_factor'];
      $rate = $this->transfer['rate'];


      if ($factor === 'divide') {
        $this->transfer['delivery_amount'] = $amount / $rate;
      } else {
        $this->transfer['delivery_amount'] =  $amount * $rate;
      }
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    }

    // $this->transfer['delivery_amount'] = $value * $this->transfer['exchange_rate'];
  }

  public function updatedTransferSenderId($value)
  {

    if ($value) {
      $sender = Agent::with('address')->find($value);
      $this->senderAgent = $sender;
      $this->senderDetails = [
        'name' => $sender?->name,
        'phone' => $sender?->phone,
        'address' => $sender?->address?->full_address,
        'balance' => 0, // $sender->getBalance($this->transfer['currency']),
      ];
    } else {
      $this->resetSenderDetails();
    }
  }
  public function updatedTransfer($value, $field)
  {
    $this->getFee();
    // $this->getTotalToReceive();
    // Recalculate when transfer data changes
    if (in_array($field, ['amount', 'receiver_fee'])) {
      //    unset($this->totalToReceive);
    }

    //  $this->calclulateDabitandCredit();
  }
  public function updatedTransferReceiverId($value)
  {
    if ($value) {
      [$source, $agentId] = explode('_', $value);
      $this->source = $source;
      $this->agentId = $agentId;
      if ($source == "internal") {
        $receiver = Agent::with('address')->find($agentId);
        $this->receiverDetails = [
          'name' => $receiver->name,
          'phone' => $receiver->phone,
          'address' => $receiver->address?->full_address,
        ];
        $this->receiverAgent = $receiver;
        $this->address = $receiver->getFullAddress();
      } else {
        $config = ApiConfig::find($source);
        $api = ApiConnection($config);
        $data = $api->getClientData($agentId);
        $this->address = $data['address'] ?? "";
      }
    }
  }

  public function CheckCanSendReceiveAgent()
  {
    if ($this->senderAgent) {
      $canSend = $this->senderAgent->canSendIn($this->transfer['currency']);
      if (!$canSend) {
        $this->addError("currency_error",   __("alerts.sender_currency_not_allowed"));
        $this->send_flag = false;
      }
    }
    if ($this->receiverAgent) {
      $canReceive = $this->receiverAgent->canReceiveIn($this->transfer['currency']);
      if (!$canReceive) {
        $this->addError("currency_error",   __("alerts.receiver_currency_not_allowed"));
        $this->send_flag = false;
      }
    }
  }


  // When currency changes
  public function updatedTransferCurrency()
  {
    $this->CheckCanSendReceiveAgent();
    $this->transfer['delivery_currency'] = $this->transfer['currency'];
  }

  public function getTotalToReceive()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    //$total = max(0, $amount - $receiverFee);
    $total = $amount;
    $this->totalToReceive = $total;
  }

  // Calculate total amount to be deducted from sender
  public function getTotalToSendProperty()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    return $amount + $senderFee;
  }

  // Reset sender details
  protected function resetSenderDetails()
  {
    $this->senderDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
      'balance' => null,
    ];
  }

  // Reset receiver details
  protected function resetReceiverDetails()
  {
    $this->receiverDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
    ];
  }
  public function calclulateDabitandCredit()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    $debit = $amount + $senderFee;
    $credit = $amount + $receiverFee;
    $revenue = $senderFee - $receiverFee;
    $this->debit = $debit;
    $this->credit = $credit;
    $this->revenue = $revenue;
  }
  public function confirmTransfer()
  {

    if (!isset($this->transfer['client_id'])) {
      $this->dispatch("sweetalert:error", ['msg' => __("alerts.client_id_required"), "title" => "Error", "type" => "error"]);
      return;
    }



    $rules = [
      'transfer.sender_id' => 'required|exists:agents,id',
      'transfer.recipient_name' => 'required',
      'transfer.recipient_phone' => 'required',
      'transfer.receiver_id' => 'required|different:transfer.sender_id',
      'transfer.currency' => 'required|exists:ledger_currencies,code',
      'transfer.amount' => 'required|numeric|min:1',
      'transfer.delivery_currency' => 'required',
      // 'transfer.send_fee' => 'required|numeric|min:0',
      'transfer.receiver_fee' => 'required|numeric|min:0',
      'transfer.transfer_date' => 'nullable|date',

    ];
    $this->transfer['secret'] = rand(1000, 9999);
    $this->transfer['reference'] = generateReferenceNumber();
    $lockKey = 'lock:transfer:' . $this->transfer['reference'];
    $lock = Cache::lock($lockKey, 10); // Lock for 10 seconds
    try {
      $this->validate($rules);
      $transfer = $this->transfer;
      $send_fee = $transfer['send_fee'] ?? 0;
      $receiver_fee = $transfer['receiver_fee'] ?? 0;

      if ($receiver_fee > $send_fee) {
        $this->dispatch("sweetalert:error", ['msg' => __("Receiver fee cannot be greater than send fee"), "title" => "Error", "type" => "error"]);
        return;
      }

      if (isset($this->transfer['id'])) {
        $this->UpdateTransfer();
        return;
      }
      if ($this->senderAgent->isitChild($this->receiverAgent->id)) {
        $this->dispatch("sweetalert:error", ['msg' => __("Cannot Transfer To Your Branch"), "title" => "Error", "type" => "error"]);
        return;
      }



      if (!$lock->get()) {
        $this->dispatch("sweetalert:error", ['msg' => __("alerts.transfer_locked"), "title" => "Error", "type" => "error"]);
        return;
      }
      try {
        (new TransferCreditLimitValidator())->validate(
          $this->senderAgent,
          $this->transfer['currency'],
          $this->transfer['amount'] + ($this->transfer['send_fee'] ?? 0)
        );
      } catch (CreditLimitExceededException $e) {
        $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
      }



      DB::beginTransaction();



      if ($this->source == "internal") {

        $this->transfer['receiver_id'] = $this->receiverAgent->id;
      }



      if ($this->source != "internal") {
        $config = ApiConfig::with("apiType")->find((int)$this->source);
        $this->transfer['receiver_id'] = $config->agent_id;
        $receiver_id = $this->agentId;

        $result = sendExternalTransaction($config, $this->transfer, $receiver_id);

        if (!$result['success']) {
          $this->dispatch("sweetalert:error", ['msg' => $result['message'], "title" => "Error", "type" => "error"]);
          DB::rollBack();
          return;
        }
        $this->transfer['api_id'] = $config->id;
        $this->transfer['external_agent_id'] = $receiver_id;
        $this->transfer['external_ref'] = $result['ref'];
        $this->transfer['secret'] = $result['password'] !== "" ? $result['password'] : rand(1000, 9999);
        $this->transfer['receiver_fee'] = $result['tax'] ?? 0;
      }

      $this->transfer['customer_details'] = $this->recipient;

      // // Map recipient data to separate fields for database storage
      // $this->transfer['recipient_phone'] = $this->recipient['recipient_phone'] ?? $this->recipient['phone'] ?? null;

      // // Map client sender data to separate fields for database storage
      $this->transfer['client_sender_name'] = $this->recipient['name'][app()->getLocale()] ?? null;
      $this->transfer['client_sender_phone'] = $this->recipient['phone'] ?? null;
      // $this->transfer['recipient_id_number'] = $this->recipient['identity_number'] ?? null;

      $service = new TransferService();

      $transfer = $service->create($this->transfer);
      $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
      $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
      event(new RefreshDataTable("transfer-table"));
      event(new RefreshDataTable("livewatch-table"));
      DB::commit();

      $this->dispatch('saveMedia', \App\Models\LedgerTransfer::class, $transfer->id);
      $address = [
        'full_address' => $this->address ?? ''
      ];
      $transfer->createOrUpdateAddress($address);


      $this->dispatch("sweetalert:success", ['msg' => __("Transfer Added Successfully"), "title" => __("Success"), "type" => "success"]);

      $this->resetForm();
      $this->dispatch("open_modal", ['action' => "hide", "id" => "MoneyTransferModal"]);
      $this->dispatch('ShowTransferReceipt', ['transfer_id' => $transfer->id]);
    } catch (ValidationException $e) {
      $firstField = array_key_first($e->validator->errors()->messages());
      $this->dispatch('focus-error', ['field' => $firstField]);

      $firstErrorMessage = $e->validator->errors()->first($firstField);
      $this->dispatch("sweetalert:error", [
        'msg' => $firstErrorMessage,
        'title' => __("alerts.error"),
        'type' => "error"
      ]);
    } catch (\Exception $e) {
      DB::rollBack();
      Log::error($e);
      $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
    } finally {
      Log::info('Transfer added', ['transfer' => $this->transfer]);
      optional($lock)->release(); // Always release the lock
    }

    $this->dispatch("UnBlockUI");
  }
  public  function UpdateTransfer()
  {
    try {
      LedgerTransfer::enableAuditing();
      $transfer = LedgerTransfer::find($this->transfer['id']);
      $this->transfer['receiver_id'] = $this->receiverAgent->id;
      $transfer->update($this->transfer);

      if (isset($transfer['external_ref'])) {
        $receiver = $transfer->receiver;
        $config = ApiConfig::with("apiType")->where("agent_id", $receiver->id)->first();
        $class = $config->apiType->class_name;
        $service_api = new $class();
        $configArray = json_decode($config->config_json, true);
        $configArray['base_url'] = $config->base_url;
        $service_api->setConfig($configArray);
        $res = $service_api->UpdateTransaction($transfer);
      } else {
        $service = new TransferService();
        $res = $service->UpdateLedger($transfer);
      }
      if ($res['success']) {
        $this->dispatch("sweetalert:success", ['msg' => $res['message'], "title" => "Success", "type" => "success"]);
        return redirect()->to('/Transfers');
      } else {
        $this->dispatch("sweetalert:error", ['msg' => $res['message'], "title" => "Error", "type" => "error"]);
      }
    } catch (\Exception $e) {
      Log::info($e);
      Log::error('Error updating transfer', ['error' => $e->getMessage()]);
    }
  }

  public function  getFee()
  {

    $sender = Agent::find($this->transfer['sender_id']);
    if ($sender) {
      $package = $sender->CurrencyTransferPackages($this->transfer['currency']);

      if ($package && $this->transfer['amount'] != 0 && isset($this->transfer['amount'])) {
        $fees = getFeePriceFromPackage($package->id, $this->transfer['amount']);
        if (!$fees['status']) {

          $this->addError('fee', $fees['msg']);
          $this->send_flag = false;
          return;
        }
        $this->send_flag = true;

        $this->transfer['send_fee'] = $fees['send_fee'];
        $this->transfer['receiver_fee'] = $fees['receive_fee'];
      } elseif ($this->transfer['amount'] != 0 && isset($this->transfer['amount'])) {
        $this->addError('fee', __("No transfer package found"));

        $this->send_flag = false;
      }
    }
  }
  public function checkBalance(): void
  {
    $sender = Agent::find($this->transfer['sender_id']);

    if (null === $sender) {
      $this->addError('balance', __('Sender Not Found'));

      return;
    }

    try {

      $total = $this->transfer['amount'] + $this->transfer['send_fee'] ?? 0;

      (new TransferCreditLimitValidator())->validate(
        $sender,
        $this->transfer['currency'] ?? '',
        $total,
        "transfer"
      );
    } catch (CreditLimitExceededException $e) {
      $this->addError('balance', $e->getMessage());
    }
  }

  // Reset the form
  public function resetForm()
  {
    $this->reset('transfer');
    $this->resetSenderDetails();
    $this->resetReceiverDetails();
    $this->recipient = [
      'name' => '',
      'phone' => '',
      'sender_identity_number' => '',
    ];

    $this->transfer['currency'] = null;
    $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
    $this->resetErrorBag();
  }
  public function getAmountInWordsProperty()
  {
    if ($this->transfer['amount'] && $this->transfer['currency']) {
      return toCurrencyWords(
        floatval($this->transfer['amount']),
        $this->transfer['currency'],
        app()->getLocale()
      );
    }
    return '';
  }
  public function getAmountInWords($amount, $currency)
  {
    if ($amount && $currency) {
      return toCurrencyWords(
        floatval($amount),
        $currency,
        app()->getLocale()
      );
    }
    return '';
  }
  public function  getBalances($agent_id)
  {
    try {
      if (!$agent_id) {
        return [];
      }
      if (str_contains($agent_id, "_")) {
        [$resource, $id] = explode("_", $agent_id);
        $agent_id = (int) $id;
      }
      Log::info("agent_id", ['agent_id' => $agent_id]);

      $agent = Agent::find($agent_id);
      $parent = $agent->rootAgent();
      return $parent->getAgentBalances();
    } catch (\Exception $e) {
      Log::error("Error getting balances", ['error' => $e->getMessage()]);
      return [];
    }
  }

  public function fieldIsVisible($key)
  {
    return $this->formFieldConfigs[$key]['visible'] ?? true;
  }

  public function fieldIsRequired($key)
  {
    return $this->formFieldConfigs[$key]['required'] ?? false;
  }

  public function fieldIsLocked($key)
  {
    return $this->formFieldConfigs[$key]['always_required'] ?? false;
  }
  public  function getFieldLabel($key)
  {
    return $this->formFieldConfigs[$key]['label'] ?? Str::title($key);
  }
  public  function getFieldTabIndex($key)
  {
    return $this->formFieldConfigs[$key]['tab_index'] ?? 1;
  }
  public function UpdatedRecipientSenderIdentityNumber()
  {
    $this->searchSender();
  }

  public function generateRandomTransferNumber(): string
  {
    $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "TRF-{$branchCode}-{$transactionId}";
  }

  public function render()
  {
    return view('livewire.transfers.add');
  }
}
