<?php

namespace App\Livewire\Transfers;

use App\Models\Agent;
use App\Models\Client;
use App\Models\IdentityType;
use App\Models\LedgerApproval;
use App\Models\LedgerTransfer;
use App\Services\Transfer\TransferService;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use OwenIt\Auditing\Events\AuditCustom;

class DeliverTransfer extends Component
{
  public ?int $transfer_id = null;
  public array $transfer = [];
  public Collection $identity_types;
  public array $accounts = [];
  public array $recipient = [];
  public ?string $secret = null;
  public bool $enable_button = true;
  public array $attachments = [];

  public string $clientsSearchUrl = '/Client/Search';
  public string $searchUrl = '/Client/Search';
  public string $transferSearchURL = '/Transfers/Search';
  public  $add_client = false;
  protected function getListeners(): array
  {
    return [
      'deliverTransfer' => 'deliverTransfer',
      "CancelAddClientEvent" => "CancelAddClientEvent",
      "AddNewClientEvent" => "AddNewClientEvent",
      "getClientData" => "getClientData",
      "deliveryReceiptClosed" => "handleDeliveryReceiptClosed",
    ];
  }
  public  function  getClientData($id)
  {
    $id = $id['id'];
    $client = Client::find($id);

    $this->recipient['recipient_name'] = $client->name;
    $this->recipient['recipient_phone'] = $client->phone;

    $this->recipient['identity_number'] = $id;
    $this->transfer['recipient_client_id'] = $client->id;
    $this->transfer['recipient_id_number'] = $client->identity_number;

    $this->add_client = false;
  }
  public function  CancelAddClientEvent()
  {
    $this->add_client = false;
  }
  public  function  AddNewClientEvent()
  {

    $this->add_client = true;
  }
  public function mount(?LedgerTransfer $transfer = null): void
  {

    $this->identity_types = IdentityType::all();

    $this->transfer = $transfer?->toArray() ?? ['id' => null];
    if ($transfer->isExternalTransfer()) {

      $this->transfer['receiver_id'] = $this->transfer["api_id"] . "_" . $this->transfer["external_agent_id"];
      $this->secret = $transfer->secret;
    } else {

      $this->transfer['receiver_id'] = "internal_" . $this->transfer["receiver_id"];
      $this->secret = $transfer->secret;
    }
    if (!empty($this->transfer['reference'])) {
      $this->searchTransfer();
    }
  }

  public function searchTransfer(): void
  {
    $reference = $this->transfer['reference'] ?? null;

    if (!$reference) {
      $this->error(__('transfers.transfer_not_found'));
      return;
    }

    $transfer = LedgerTransfer::with(['client', 'sender', 'receiver'])
      ->where('reference', $reference)
      ->orWhere('id', $reference)
      ->first();

    if (!$transfer) {
      $this->error(__('transfers.transfer_not_found'));
      return;
    }

    $this->transfer = $transfer->toArray();
    $this->transfer['amount_number'] = NoStylePriceFormat($this->transfer['delivery_amount'], $this->transfer['delivery_currency']);

    $this->transfer['written_amount'] = toCurrencyWords($this->transfer['delivery_amount'], $this->transfer['delivery_currency']);

    if ($transfer->status === 'Completed') {
      $this->error(__('transfers.transfer_already_delivered'));
      $this->enable_button = false;
    }

    $this->accounts = $transfer->type !== 'transfer'
      ? LedgerApproval::GetDestinationAgents()->toArray()
      : LedgerTransfer::GetDestinationAgents('delivering')->toArray();

    $this->dispatch('HideClientCard', [
      'show' => $transfer->type === 'transfer',
    ]);

    $this->dispatch('refresh-select2', [
      'target' => 'transfer.receiver_id',
      'options' => $this->formatAccountOptions($this->accounts),
    ]);
  }

  public function searchReceiver(): void
  {
    $this->validate([
      'recipient.identity_number' => 'required|string',
    ]);

    $identityNumber = $this->recipient['identity_number'] ?? null;

    $client = Client::query()
      ->where('phone', $identityNumber)
      ->orWhere('identity_number', $identityNumber)
      ->orWhere('name', $identityNumber)
      ->orWhere('id', $identityNumber)
      ->first();

    if (!$client) {
      $this->error(__('transfers.client_not_found'));
      return;
    }

    $this->recipient = array_merge($client->toArray(), [
      'sender_client_id' => $client->id,
      'sender_identity_number' => $client->identity_number,
    ]);

    $this->transfer['recipient_client_id'] = $client->id;
    $this->transfer['recipient_id_number'] = $client->identity_number;

    $this->dispatch('LoadClientData', ['id' => $client->id]);
  }

  public function deliverTransfer(): void
  {
    $transfer = LedgerTransfer::where('reference', $this->transfer['reference'] ?? null)->first();

    if (!$transfer || empty($this->transfer['id'])) {
      $this->error(__('transfers.transfer_not_found'));
      return;
    }

    if (in_array($transfer->status, ['reverse', 'cancelled'], true)) {
      $this->error(__('transfers.transfer_already_cancelled'));
      return;
    }

    if ($transfer->type === 'approval') {
      $rules = LedgerApproval::getDeliveryRules();
    } else {
      if ($transfer->status === 'completed') {
        $this->error(__('transfers.transfer_already_delivered'));
        return;
      }
      $rules = LedgerTransfer::getDeliveryRules();

      if (empty($this->transfer['recipient_client_id'])) {
        $this->error(__('transfers.recipient_not_found'));
        return;
      }
    }

    $this->validate($rules);

    if ($this->secret !== $transfer->secret) {
      $this->error(__('transfers.invalid_secret'));
      return;
    }

    try {
      $this->processDelivery($transfer);
      $this->success(__('transfers.transfer_delivered_successfully'));
      $this->dispatch("RefreshYajraDatatable", ['table' => 'transfer-table']);
    } catch (\Throwable $e) {
      Log::info($e);
      $this->error($e->getMessage());
    }
  }

  private function processDelivery(LedgerTransfer $transfer)
  {
    $receiver_id = preg_replace('/\D/', '', $this->transfer['receiver_id']);
    $transfer->recipient_client_id = $this->recipient['sender_client_id'] ?? null;
    $transfer->recipient_id_number = $this->recipient['sender_identity_number'] ?? null;
    $transfer->receiver_id = $receiver_id;
    $transfer->disableAuditing();
    $transfer->receiver_id = $transfer->receiver->rootAgent()->id;

    (new TransferService())->DeliverTransfer($transfer);
    $transfer->delivered_at = now();
    $transfer->status = 'Completed';

    $transfer->save();
    $this->dispatch("ShowDeliveryReceipt", ['transfer_id' => $transfer->id]);
    $this->logAudit($transfer);

    $this->dispatch('saveMedia', LedgerTransfer::class, $transfer->id);
    $this->dispatch('refreshYajraDatatable', ['table' => 'transfer-table']);
    $this->dispatch('refreshYajraDatatable', ['table' => 'approval-table']);
  }

  private function logAudit(LedgerTransfer $transfer): void
  {
    $recipient = Client::find($transfer->recipient_client_id);

    $transfer->enableAuditing();
    $transfer->auditEvent = 'Delivered';
    $transfer->isCustomEvent = true;
    $transfer->auditCustomOld = [];
    $transfer->auditCustomNew = [
      'recipient_client_id' => $transfer->recipient_client_id,
      'recipient_id_number' => $transfer->recipient_id_number,
      'recipient_name' => $recipient->name ?? '',
      'recipient_phone' => $recipient->phone ?? '',
      'status' => 'Completed',
    ];

    Event::dispatch(new AuditCustom($transfer));
    $transfer->disableAuditing();
  }

  private function formatAccountOptions(array $accounts): array
  {
    return array_map(function ($account) {
      $name = $account['name'] ?? '';
      return [
        'id' => $account['id'],
        'text' => $account['text'] ?? $name,
      ];
    }, $accounts);
  }

  private function error(string $message): void
  {
    $this->dispatch('sweetalert:error', [
      'msg' => $message,
      'title' => 'Error',
      'type' => 'error',
    ]);
  }

  private function success(string $message): void
  {
    $this->dispatch('sweetalert:success', [
      'msg' => $message,
      'title' => 'Success',
      'type' => 'success',
    ]);
  }

  public function updatedRecipientIdentityNumber(): void
  {
    $this->searchReceiver();
  }

  public function updatedTransferReference(): void
  {
    $this->searchTransfer();
  }

  public function handleDeliveryReceiptClosed()
  {
    // Redirect back to transfers list after delivery receipt modal is closed
    return redirect('/Transfers');
  }

  public function render(): View
  {
    return view('livewire.transfers.deliver-transfer');
  }
}
