<?php

namespace App\Livewire\User\View;

use App\Livewire\Components\Contract;
use App\Models\User;
use Carbon\Carbon;
use Livewire\Component;

class Dahboard extends Component
{
    public $tenantId;
    public $tenant;
    public $statistics = [];
    public $timeline = [];

    public function mount()
    {


        $this->loadStatistics();
        $this->prepareChartData();
        $this->emitPaidVsUnpaidChart();
        $this->loadInvoiceTimeline();
        $this->dispatchLeaseBreakdownChart();
    }

    public function loadStatistics()
    {
        $tenant = $this->tenant;
        // Fetch leases for tenant
        $this->statistics = [
            'total_rent_paid'     => $tenant->getTotalPayments(),
            'outstanding_balance' => $tenant->OutstandingInvoicesAmount(),
            'next_due_date'       => optional(
                $tenant->getPendingInvoices()->first()
            )?->due_date?->format('Y-m-d'),
            'lease_start'         => optional($tenant->contracts->sortByDesc('start_date')->first())?->start_date?->format('Y-m-d'),
            'lease_end'           => optional($tenant->contracts->sortByDesc('start_date')->first())?->end_date?->format('Y-m-d'),
            'total_late_fees'     => $tenant->invoices->sum('late_fee'),
        ];
    }
    public function dispatchLeaseBreakdownChart()
    {
        $breakdown = $this->tenant->contracts->map(function ($lease) {
            $invoices = $lease->invoices()->sum('total_amount');
            $payments = $lease->invoices()->with('payments')->get()->flatMap->payments->sum('amount');
            return [
                'name' => $lease->property->name ?? 'Lease #' . $lease->id,
                'invoices' => round($invoices, 2),
                'payments' => round($payments, 2),
                'balance' => round($invoices - $payments, 2),
            ];
        });

        $labels = $breakdown->pluck('name');
        $series = [
            ['name' => __("general.invoices"), 'data' => $breakdown->pluck('invoices')],
            ['name' => __("general.payments"), 'data' => $breakdown->pluck('payments')],
            ['name' => __("general.balance"), 'data' => $breakdown->pluck('balance')],
        ];

        $this->dispatch('InitializeLeaseBreakdownChart', [

            'data' => [
                'labels' => $labels,
                'series' => $series,
            ]

        ]);
    }


    public function loadInvoiceTimeline()
    {
        $this->timeline = $this->tenant->invoices()
            ->with(['payments' => fn($q) => $q->latest()])
            ->orderBy('issue_date', 'desc')->limit(4)
            ->get()
            ->map(function ($invoice) {
                $latestPayment = $invoice->payments->first();

                return [
                    'title' => 'Invoice #' . $invoice->id,
                    'description' => $invoice->status === 'paid'
                        ? 'Paid via ' . ($latestPayment->method ?? 'Unknown')
                        : 'Awaiting payment',
                    'issue_date' => $invoice->issue_date->diffForHumans(),
                    'point_class' => match ($invoice->status) {
                        'paid' => 'timeline-point-success',
                        'pending' => 'timeline-point-warning',
                        default => 'timeline-point-secondary',
                    },
                    'status' => ucfirst($invoice->status),
                    'payment_date' => optional($latestPayment)->payment_date?->format('Y-m-d') ?? null,
                ];
            })->toArray();
    }
    protected function prepareChartData()
    {
        $monthlyData = [];

        // Initialize past 12 months with zero values
        for ($i = 11; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i)->format('Y-m');
            $monthlyData[$month] = 0;
        }

        // Aggregate tenant payments per month
        foreach ($this->tenant->payments as $payment) {
            $month = Carbon::parse($payment->created_at)->format('Y-m');
            if (isset($monthlyData[$month])) {
                $monthlyData[$month] += $payment->amount;
            }
        }

        $labels = array_keys($monthlyData);
        $series = [
            [
                'name' => 'Total Paid',
                'data' => array_values($monthlyData),
            ]
        ];
        $data = [
            'labels' => $labels,
            'series' => $series
        ];

        // Emit the JS event exactly as expected
        $this->dispatch('InitializePaymentChart', ["data" => $data]);
    }
    public function emitPaidVsUnpaidChart()
    {
        $totalPaid = $this->tenant->getTotalPayments(); // total payment amounts
        $totalDue = $this->tenant->invoices()->sum('total_amount'); // full invoice amounts

        $unpaid = max($totalDue - $totalPaid, 0); // avoid negative numbers

        $series = [(float)$totalPaid, (float)$unpaid];
        $labels = ['Paid', 'Unpaid'];
        $data = [
            'series' => $series,
            'labels' => $labels,
        ];
        $this->dispatch('InitializePaidVsUnpaidChart', ['data' => $data]);
    }
    public function render()
    {
        return view('livewire.user.view.dahboard');
    }
}
