<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Translatable\HasTranslations;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\LedgerAccountController;
use Abivia\Ledger\Messages\Account;
use Abivia\Ledger\Messages\Create;
use Abivia\Ledger\Models\LedgerAccount;
use App\Interfaces\Avatar;
use App\Traits\HasAvatar;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\Builder;

use Illuminate\Database\Eloquent\Scope;
use Illuminate\Support\Facades\Cache;

class Currency extends Model implements Avatar, HasMedia
{
  use HasFactory;
  protected $fillable = ['name', 'symbol', 'code', 'exchange_rate', 'is_default', 'color', 'rate_factor', 'order_id'];
  use HasTranslations;
  use HasAvatar;
  use InteractsWithMedia;
  public $translatable = ['name'];
  protected static function booted()
  {
    static::addGlobalScope('ordered', function (Builder $builder) {
      $builder->orderBy('order_id');
    });
  }
  public function ledgerCurrency()
  {
    return $this->hasOne(\Abivia\Ledger\Models\LedgerCurrency::class, 'code', 'code');
  }
  public function getDefaultAvatar(): string
  {
    return $this->avatar ?? 'https://ui-avatars.com/api/?name=' . urlencode($this->code) . '&background=685dd8'   . '&color=fff';
  }
  public static function getCurrencyName($code)
  {
    return cache()->remember("currency_name_{$code}", 60 * 60, function () use ($code) {
      $currency = Currency::where('code', $code)->first();
      if (isset($currency) && ($currency->name !== null && $currency->name !== "")) return $currency->name;
      return  $code;
    });
  }
  public static function getCurrencyColor($code)
  {
    return Cache::remember("currency_color_{$code}", 60 * 60, function () use ($code) {
      $currency = Currency::where('code', $code)->first();
      if (isset($currency) && ($currency->color !== null && $currency->color !== "")) return $currency->color;
      return  $code;
    });
  }
  public function createLedgerAccountsForCurrency(): bool
  {
    $currency = $this;
    try {
      $controller = new LedgerAccountController();

      // Main currency code (like 1410, 1420...)
      $currencyCode = generateCurrencyAccountCode('1400', $currency->id);
      $buyCode = $currencyCode + 1;
      $sellCode = $currencyCode + 2;

      $accounts = [
        // Root currency exchange account
        // [
        //     'code' => (string)$currencyCode,
        //     'names' => [
        //         ['language' => 'en', 'name' => 'Exchange ' . $currency->name ?? $currency->code],
        //         ['language' => 'ar', 'name' =>  'صرف ' .  $currency->name ?? $currency->code]
        //     ],
        //     'parent' => '1400',
        //     'debit' => true,
        //     'category' => true,
        // ],
        // Buy account
        [
          'code' => (string)$buyCode,
          'names' => [
            ['language' => 'en', 'name' => $currency->code . 'Buy'],
            ['language' => 'ar', 'name' => $currency->code . 'شراء']
          ],
          'parent' => (string)$currencyCode,
          'debit' => true,
          'category' => false,
        ],
        // Sell account
        [
          'code' => (string)$sellCode,
          'names' => [
            ['language' => 'en', 'name' => $currency->code . 'Sell'],
            ['language' => 'ar', 'name' => $currency->code . 'بيع']
          ],
          'parent' => (string)$currencyCode,
          'debit' => true,
          'category' => false,
        ],
      ];

      foreach ($accounts as $account) {
        $response = $controller->add(Account::fromArray(
          array_merge($account, [
            'reference' => $currency->ledger_reference_uuid ?? null,
            'extra' => json_encode(['currency_id' => $currency->id])
          ])
        ));
      }

      return true;
    } catch (Breaker $e) {
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true),
        'validation_rules' => config('ledger.rules'),
      ]);
    } catch (\Exception $e) {
      Log::error('Currency account creation failed: ' . $e->getMessage());
      return false;
    }
  }
  public function exchangeAccounts()
  {
    return $this->hasMany(LedgerAccount::class, 'extra->currency_id', 'id');
  }

  public  function GetExchangeAccountCode(): array
  {

    $data = [
      'exchange' =>  generateCurrencyAccountCode('1400', $this->id),
      'buy' =>  generateCurrencyAccountCode('1400', $this->id),
      'sell' =>   generateCurrencyAccountCode('1400', $this->id)
    ];

    return $data;
  }
  public static function getExchangeRate($currencyCode): float
  {
    $currency = Currency::where('code', $currencyCode)->first();
    if ($currency) {
      return $currency->exchange_rate;
    }
    return 1.0; // Default exchange rate if not found
  }
  public static function getRateFactor($currencyCode)
  {
    $currency = Currency::where('code', $currencyCode)->first();
    if ($currency) {
      return $currency->rate_factor;
    }
    return "multiply"; // Default rate factor if not found
  }
  public static function getCurrencyShortcuts(): array
  {
    $currencies = self::all();
    $shortcuts = [];

    foreach ($currencies as $from) {
      foreach ($currencies as $to) {
        if ($from->id !== $to->id) {
          $shortcuts[] = [
            'from_code' => $from->code,
            'from_name' => $from->getTranslation('name', app()->getLocale()),
            'to_code'   => $to->code,
            'to_name'   => $to->getTranslation('name', app()->getLocale()),
          ];
        }
      }
    }

    return $shortcuts;
  }
  public  function  getCurrencyExchangeBalances()
  {
    $exchangeAccounts = $this->exchangeAccounts()->with("balances")->get();

    if ($exchangeAccounts->isEmpty()) {
      return 0; // No exchange accounts found
    }

    // Get the first exchange account (assuming one per currency)
    $account = $exchangeAccounts->first();

    // Find the balance for this specific currency
    $balance = $account->balances->firstWhere('currency', $this->code);

    return $balance ? (float) $balance->balance : 0;
  }
}
