<?php

namespace App\Models;

use Abivia\Ledger\Http\Controllers\ReportController;
use Illuminate\Database\Eloquent\Model;

use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Models\LedgerBalance;
use Abivia\Ledger\Models\LedgerDomain;
use Illuminate\Support\Facades\DB;
use Abivia\Ledger\Reports\TrialBalanceReport;
use Abivia\Ledger\Messages\Report;
use Illuminate\Database\Eloquent\SoftDeletes;

class CustomLedgerAccount extends LedgerAccount
{
  protected $table = 'ledger_accounts';
  use SoftDeletes;
  protected $casts = [
    'extra' => 'array', // لتحويل JSON إلى array تلقائياً
  ];
  protected $appends = ['type'];
  public function parent()
  {
    return $this->belongsTo(self::class, 'parentUuid', 'ledgerUuid');
  }

  public function children()
  {
    return $this->hasMany(self::class, 'parentUuid', 'ledgerUuid');
  }

  public function scopeForAgent($query, $agentId)
  {
    return $query->whereJsonContains('extra->agent_id', $agentId);
  }

  public function agent()
  {
    return $this->belongsTo(Agent::class, 'agent_id', 'id');
  }
  public function getAgentBalances(): \Illuminate\Support\Collection
  {
    return LedgerBalance::query()
      ->select(
        'ledger_balances.currency',
        DB::raw('SUM(CAST(ledger_balances.balance AS DECIMAL(15,2))) as balance'),
        DB::raw('COUNT(DISTINCT ledger_balances.ledgerUuid) as account_count')
      )
      ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
      ->where('ledger_accounts.code',   $this->code)
      ->groupBy('ledger_balances.currency')
      ->get()
      ->map(function ($item) {
        return [
          'currency' => $item->currency,
          'balance' => (float) $item->balance,
          'account_count' => $item->account_count,
          'formatted_balance' => number_format(abs($item->balance), 2) . ' ' . $item->currency,
          'abs_balance' => abs($item->balance)
        ];
      });
  }



  public function getBalanceByDate($date, $currency)
  {
    $accountCode = $this->code;

    $report = Report::fromArray([
      'name'     => 'trialBalance',
      'currency' => $currency,
      'toDate'   => $date,
      'options'  => [
        'format'   => 'raw',
        'depth'    => 3,
        'language' => [app()->getLocale()],
      ],
    ]);

    $reporter = new ReportController();
    $reportData = $reporter->generate($report);

    // Make sure accounts exist


    $accountRow = $reportData
      ->firstWhere('code', $accountCode);

    // Use 'total' for the final as-of balance
    return $accountRow ? (float) $accountRow->total : 0.0;
  }
  public function getTypeAttribute()
  {
    return $this->credit == 1 ? 'credit' : 'debit';
  }
}
