<?php

namespace App\Models;

use Abivia\Ledger\Models\JournalEntry;
use Abivia\Ledger\Models\LedgerAccount;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Rawilk\Settings\Facades\Settings;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Support\Arr;
use App\Traits\CustomAuditable;
use App\Traits\TransfersTrait;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use OwenIt\Auditing\Contracts\Auditable;
use Laravel\Scout\Searchable;

class LedgerBooking extends Model implements HasMedia, Auditable
{
  protected $table = 'ledger_transfers';
  use SoftDeletes;
  use TransfersTrait;
  use Searchable;

  public $meilisearchSettings = [
    'filterableAttributes' => [
      '__soft_deleted',
      'status',
      'currency',
      'sender_id',
      'receiver_id',
      'type',
      'recipient_name',
      'recipient_phone',
      'amount',
      'delivery_amount',
      'send_fee',
      'receiver_fee',
      'created_at',
      'delivered_at',
      'notes',
      'receiver_note',
    ],
    'sortableAttributes' => [
      'amount',
      'delivery_amount',
      'send_fee',
      'receiver_fee',
      'created_at',
      'delivered_at',
    ],
    'searchableAttributes' => [
      'reference',
      'sender_name',
      'receiver_name',
      'recipient_name',
      'recipient_phone',
      'recipient_id_number',
      'sender_phone',
      'sender_id_number',
      'receiver_phone',
      'receiver_id_number',
      'notes',
      'receiver_note',
    ],
    'rankingRules' => [
      'words',
      'typo',
      'proximity',
      'attribute',
      'exactness',

    ],
  ];
  public function toSearchableArray()
  {
    return [
      'id' => $this->id,
      'reference' => $this->reference,
      'sender_name' => $this->sender->name ?? "",
      'receiver_name' => $this->receiver->name ?? "",
      'amount' => $this->amount,
      'currency' => $this->currency,
      'status' => $this->status,
      'recipient_phone' => $this->recipient_phone,
      'recipient_id_number' => $this->recipient_id_number,
      "recipient_name" => $this->recipient_name,

      "sender_phone" => $this->sender->phone ?? "",
      "sender_id_number" => $this->sender->id_number ?? "",
      "sender_account_code" => $this->sender->account_code ?? "",

      "receiver_phone" => $this->receiver->phone ?? "",
      "receiver_id_number" => $this->receiver->id_number ?? "",
      "receiver_account_code" => $this->receiver->account_code ?? "",
      'notes' => $this->notes,
      'receiver_note' => $this->receiver_note,
      'deleted_at' => $this->deleted_at,
      '__soft_deleted' => $this->deleted_at != null,
    ];
  }

  /**
   * Determine if the model should be searchable.
   * Include soft-deleted records in search results.
   */
  public function shouldBeSearchable()
  {
    // Include both active and soft-deleted records in search
    return true;
  }
  protected static function booted()
  {
    static::addGlobalScope('booking', function ($query) {
      $query->where('type', 'Booking')->orwhere("type", "exchange");
    });

    static::addGlobalScope('agentTransfers', function (Builder $builder) {
      if (Auth::guard('agent')->check() && Auth::guard("agent")->user()->agent_id != null) {

        $agent = Auth::guard('agent')->user()->agent->rootAgent();
        $agentId = $agent->id;
        $childs = $agent->getallchildid();

        $builder->where(function ($query) use ($childs) {
          $query->whereIn('sender_id', $childs)
            ->orWhereIn('receiver_id', $childs);
        });
      }
    });
  }

  use HasFactory;
  use InteractsWithMedia;
  use \OwenIt\Auditing\Auditable, CustomAuditable;


  protected $types = [
    'transfer' => ['template' => 'vertical'],
    'data' => [['sender_id' => 'select'], 'receiver_id', 'amount', 'currency', 'receiver_fee', 'sender_fee', 'fee', 'status', 'reference'],
    'record' => ['sender_id', 'reciever_id']
  ];

  protected $filters = [
    'sender_id',
    'receiver_id',
    'type',
    'status',
    'created_at',
    'delivered_at',
    'currency',
    'from_date',
    'to_date',


  ];
  protected $fillable = [
    'id',
    'receiver_fee',

    'sender_id',
    'receiver_id',
    'amount',
    'delivery_amount',
    'fee',
    'currency',
    'status',
    'reference',
    'ledger_journal_uuid',
    'customer_details',
    'id',
    'debit_journal_uuid',
    'credit_journal_uuid',
    'client_id',
    'send_fee',
    'delivery_currency',
    'notes',
    'secret',
    'type',
    'date',
    'recipient_name',
    'recipient_client_id',
    'recipient_phone',
    'recipient_id_number',
    'client_sender_name',
    'client_sender_phone',
    'client_sender_id_number',
    'sender_account_code',
    'receiver_account_code',
    'user_id',
    'rate',
    'rate_factor',
    'extra',
    'transfer_id',
    'receiver_note'




  ];
  protected $appends = ['status_color'];
  protected $statusColors = [
    'pending' => 'warning',
    'approved' => 'success',
    'rejected' => 'danger',
    'cancelled' => 'danger',
    'completed' => 'success',
    'reverse' => 'danger',
    'deleted' => 'danger',
    'booking' => 'info',
  ];

  protected $casts = [
    'customer_details' => 'array',
    'extra' => 'array',
    'created_at' => 'datetime:Y-m-d H:i:s',
    'delivered_at' => 'datetime:Y-m-d H:i:s'
  ];
  public  function getStatusAttribute($value)
  {
    return $value; // == "completed" ? 'booking' : $value;
  }
  public function reason()
  {
    return $this->belongsTo(Reason::class);
  }
  public function sender()
  {
    return $this->belongsTo(Agent::class, 'sender_id', 'id');
  }

  public function receiver()
  {
    return $this->belongsTo(Agent::class, 'receiver_id', 'id');
  }
  public function debitJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'debit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public function creditJournalEntry()
  {
    return $this->belongsTo(\Abivia\Ledger\Models\JournalEntry::class, 'credit_journal_uuid', 'journalEntryId')
      ->with('details');
  }

  public  function getStatusColorAttribute()
  {
    return $this->statusColors[$this->status];
  }
  public  function  client()
  {
    return $this->belongsTo(Client::class);
  }



  public static function getFilterableFields()
  {

    return collect([
      __('transfers.sender_agent') => ["key" => "sender_id", 'type' => 'x-Mulk.select2', 'options' =>  Agent::pluck('name', 'id')->toArray()],
      __('transfers.receiver_agent') => ["key" => "receiver_id", 'type' => 'x-Mulk.select2', 'options' => Agent::pluck('name', 'id')->toArray()],
      __("transfers.from_date") => [
        'key' => 'from_date',
        'type' => 'date',
        'options' => []
      ],
      __("transfers.to_date") => [
        'key' => 'to_date',
        'type' => 'date',
        'options' => []
      ],


    ])
      ->filter(fn($config) => is_array($config) && isset($config['type']))
      ->toArray();
  }
  public function getData($queryBuilder, array $filters = [], $type = "data", $datatype = "all")
  {


    foreach ($filters as $filter => $value) {
      if (!in_array($filter, $this->filters)) {
        continue;
      }
      if ($filter === 'from_date' && $value) {
        $queryBuilder->whereDate('created_at', ">=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'to_date' && $value) {
        $queryBuilder->whereDate('created_at', "<=", Carbon::parse($value)->format("Y-m-d"));
      } elseif ($filter === 'delivered_at' && is_array($value) && count($value) === 2) {
        $value["start"] = Carbon::parse($value["start"])->format("Y-m-d");
        $value["end"] = Carbon::parse($value["end"])->format("Y-m-d");
        $queryBuilder->whereDate($filter, ">=", $value["start"])->whereDate($filter, "<=", $value["end"]);
      } else {
        if (isset($value) && $value != "") {

          $queryBuilder = $this->getquery($queryBuilder, $filter, $value);
        }
      }
    }
    if ($type == "data") {

      return $queryBuilder;
    } else
      return $queryBuilder->count();
  }
  public  function getquery($query, $key, $value)
  {



    $query = $query->where($key, $value);

    return $query;
  }
  public  function DeliveredCurrency()
  {
    return $this->belongsTo(Currency::class, 'delivery_currency', 'code');
  }

  public function SenderCurrency()
  {
    return $this->belongsTo(Currency::class, 'currency', 'code');
  }
  public function scopeFilter($query, $filters)
  {
    $query = $this->getData($query, $filters);

    return $query;
  }
  public function ScopeBookingTransfer($query)
  {
    return $query->where('type', 'booking');
  }
  public function ScopeTransfer($query)
  {
    return $query->where('type', 'transfer');
  }
  public function ScopeExchange($query)
  {
    return $query->where('type', 'exchange');
  }

  public function getLedgerJournal()
  {
    return $this->belongsTo(JournalEntry::class, 'ledger_journal_uuid', 'journalEntryId');
  }
  public function user()
  {
    return $this->belongsTo(User::class);
  }
  public  function getParsedReceipt()
  {
    try {
      $type = $this->type;
      $template_id = settings::get("default_template.$type");
      $template = Template::find($template_id);
      $html = $template->content;
      $companyName = Settings::get('general.company_name');
      $logo =  '/storage/Logo/' . Settings::get('logo', null);
      return str_replace([
        '%company_name%',
        '%logo%',
        '%date%',
        '%receiver_id%',
        '%sender_id%',
        '%reference%',
        '%client_name%',
        '%client_phone%',
        '%secret%',
        '%identity_number%',
        '%notes%',
        '%amount%',
        '%send_fee%',
        '%receive_fee%',
        '%delivered_at%',
        '%receiver_address%',
        '%sender_address%',



      ], [
        $companyName, // Or get from DB
        "<img src=\"$logo\" alt=\"Logo\" class=\"img-fluid mt-2 embed-responsive\" style=\"max-width: 100px; max-height: 100px;\" />",
        $this->created_at->format('Y-m-d H:i:s'),
        $this->receiver->name ?? '',
        $this->sender->name ?? '',
        $this->reference,
        $this->client->name,
        $this->client->phone,
        $this->secret,
        $this->identity_number,
        $this->notes,
        format_money($this->amount, "USD"),
        format_money($this->send_fee, "USD"),
        format_money($this->receive_fee, "USD"),
        $this->delivered_at,
        $this->receiver->getFullAddress() ?? '',
        $this->sender->getFullAddress() ?? '',


      ], $html);
    } catch (\Exception $e) {
      return "الرجاء تعيين  قالب  لاشعار الحوالة ";
    }
  }
  public function transformAudit(array $data): array
  {
    if (Arr::has($data, 'new_values.sender_id')) {

      $data['new_values']['sender_id'] =  $this->sender->name;
    }
    if (Arr::has($data, 'new_values.receiver_id')) {

      $data['new_values']['sender_id'] =  $this->receiver->name;
    }
    if (Arr::has($data, 'new_values.amount')) {

      $data['new_values']['amount'] = format_money($this->amount, $this->currency);
    }
    if (Arr::has($data, 'new_values.delivery_amount')) {

      $data['new_values']['delivery_amount'] = format_money($this->delivery_amount, $this->delivery_currency);
    }




    return $data;
  }
  public  static function GetSourceAgents($type = "booking")
  {
    if (auth()->guard('agent')->check()) {
      $rootagent = auth()->guard('agent')->user()->agent->rootAgent();
      return Agent::IsCashier()->where("parent_id", $rootagent->id)->get();
    }
    if ($type == "collection")
      return self::GetUserAgent();

    return Agent::IsAgent()->get();
  }
  public  static function GetDestinationAgents($type = "booking")
  {

    if (auth()->guard('agent')->check()) {
      $rootagent = auth()->guard('agent')->user()->agent->rootAgent();
      return Agent::IsCashier()->where("parent_id", $rootagent->id)->get();
    }
    if ($type == "payment")
      return self::GetUserAgent();
    return Agent::IsAgent()->get();
  }
  public  static function GetUserAgent()
  {
    $user = Auth::user();
    return $user->agents;
  }
  public function journalEntries()
  {
    return $this->hasMany(JournalEntry::class, 'transfer_id', 'id');
  }
  public function isReversed()
  {
    return   $this->status == "reverse";
  }
}
