<?php

namespace App\Rules;

use App\Models\Agent;
use Illuminate\Contracts\Validation\ValidationRule;
use Closure;

class NotChildOf implements ValidationRule
{
  protected $parentId;
  protected $parentField;

  public function __construct($parentId, $parentField = 'parent_id')
  {
    $this->parentId = $parentId;
    $this->parentField = $parentField;
  }

  public function validate(string $attribute, mixed $value, Closure $fail): void
  {
    if (!$value || !$this->parentId) {
      return;
    }

    $child = Agent::find($value);
    $parent = Agent::find($this->parentId);

    if (!$child || !$parent) {
      return;
    }

    // Check direct parent-child relationship
    if ($child->{$this->parentField} == $this->parentId) {
      $fail('The selected agent cannot be a child of the other agent.');
      return;
    }

    // Check multi-level hierarchy if needed
    if ($this->isDescendantOf($child, $parent)) {
      $fail('The selected agent cannot be a child of the other agent.');
      return;
    }
  }

  private function isDescendantOf($child, $parent)
  {
    $current = $child;

    while ($current && $current->{$this->parentField}) {
      if ($current->{$this->parentField} == $parent->id) {
        return true;
      }
      $current = Agent::find($current->{$this->parentField});
    }

    return false;
  }
}
