<?php

namespace App\Services\ExchangeRates;

use App\Models\Currency;
use App\Models\ExchangePrice;
use App\Services\ExchangeRates\ExchangePriceResolver;
use Illuminate\Support\Facades\Log;

class CurrencyConverter
{
    public function convert($amount, $fromCurrency, $toCurrency, $currencies, $type = "balance", $manualRate = null, $manualOperation = null)
    {
        if ($fromCurrency === $toCurrency) {
            return $amount;
        }
        Log::info("amount: " . $amount . " fromCurrency: " . $fromCurrency . " toCurrency: " . $toCurrency . " manualRate: " . $manualRate . " manualOperation: " . $manualOperation);

        // If manual override exists, use it immediately:
        if ($manualRate !== null && in_array($manualOperation, ['*', '/'])) {
            Log::info("Using manual rate: " . $manualRate . " operation: " . $manualOperation);
            if ($manualOperation === '*') {
                return round($amount * $manualRate, 2);
            } else {
                if ($manualRate == 0) {
                    throw new \Exception("Division by zero in currency conversion.");
                }
                return round($amount / $manualRate, 2);
            }
        }

        // Otherwise try to find ExchangePrice in DB
        $exchangePrice = ExchangePrice::where('from_currency', $fromCurrency)
            ->where('to_currency', $toCurrency)
            ->where('type', $type)
            ->first();

        if ($exchangePrice) {
            $resolver = new ExchangePriceResolver();
            $rate = $resolver->resolve($exchangePrice);

            if ($exchangePrice->operation === '*') {
                return round($amount * $rate, 2);
            } elseif ($exchangePrice->operation === '/') {
                if ($rate == 0) {
                    throw new \Exception("Division by zero in currency conversion.");
                }
                return round($amount / $rate, 2);
            } else {
                throw new \Exception("Unsupported operation '{$exchangePrice->operation}'");
            }
        }


        // Fallback to currency-based logic
        if ($type === "balance") {
            return $this->fallbackCurrencyConvert($amount, $fromCurrency, $toCurrency, $currencies);
        }
        return null;
    }


    protected function fallbackCurrencyConvert($fromAmount, $fromCurrency, $toCurrency, $currencies): float
    {
        $default = collect($currencies)->firstWhere('is_default', 1);
        $defaultCode = $default['code'];

        if ($fromCurrency === $defaultCode) {
            $toData = collect($currencies)->firstWhere('code', $toCurrency);
            return $this->exchange($fromAmount, $toData['exchange_rate'], $toData['rate_factor'], 'reverse');
        }

        if ($toCurrency === $defaultCode) {
            $fromData = collect($currencies)->firstWhere('code', $fromCurrency);
            return $this->exchange($fromAmount, $fromData['exchange_rate'], $fromData['rate_factor'], 'normal');
        }

        // 3-way conversion: from → default → to
        $fromData = collect($currencies)->firstWhere('code', $fromCurrency);
        $toData = collect($currencies)->firstWhere('code', $toCurrency);

        $toDefault = $this->exchange($fromAmount, $fromData['exchange_rate'], $fromData['rate_factor'], 'normal');
        return $this->exchange($toDefault, $toData['exchange_rate'], $toData['rate_factor'], 'normal');
    }

    protected function exchange($amount, $rate, $factor, $direction = 'normal')
    {
        try {
            Log::info("factor" . $factor . " amount: " . $amount . " rate: " . $rate . " direction: " . $direction);
            if ($direction === 'normal') {
                if ($factor == "multiply") {
                    return $amount * $rate;
                } else {
                    return $amount / $rate;
                }
            } else {
                if ($factor == "multiply") {
                    return $amount / $rate;
                }
                return $amount  * ($rate);
            }
        } catch (\Exception $e) {
            Log::info("amount" . $amount . " rate: " . $rate . " factor: " . $factor . "Exception: " . $e);
            Log::info($e);
        }
    }
}
