<?php
// app/Services/ExchangeRates/ExchangeRateManager.php
namespace App\Services\ExchangeRates;

use App\Models\ExchangeRate;
use App\Services\ExchangeRates\Contracts\ExchangeRateProvider;
use App\Services\ExchangeRates\Providers\OpenExchangeRatesProvider;
use App\Services\ExchangeRates\Providers\CurrencyLayerProvider;
use App\Services\ExchangeRates\Providers\HaramAltinProvider;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;

class ExchangeRateManager
{
    protected ExchangeRateProvider $provider;
    protected int $cacheSeconds = 60; // cache duration
    public function __construct(string $providerKey = null)
    {
        $this->provider = $this->resolveProvider($providerKey ?? config('Orange.exchange.default'));
    }

    protected function resolveProvider(string $key): ExchangeRateProvider
    {
        return match ($key) {
            'openexchange' => new OpenExchangeRatesProvider(),
            'currencylayer' => new OpenExchangeRatesProvider(),
            'haramaltin' => new HaramAltinProvider(),
            default => throw new \Exception("Unsupported exchange rate provider: $key")
        };
    }
    public function getProvider(): ExchangeRateProvider
    {
        return $this->provider;
    }
    public function getRate(string $from, string $to, ?string $apiKey = null): float
    {
        // 1. Try from DB
        $rate = ExchangeRate::where('from_currency', $from)
            ->where('to_currency', $to)
            ->latest('rate_date')
            ->first();

        if ($rate) {
            return $rate->rate;
        }

        // 2. Fallback to API
        $fetchedRate = $this->provider->getRate($from, $to, $apiKey);

        if (!$fetchedRate) {
            throw new \Exception("Unable to fetch exchange rate from API.");
        }

        // 3. Save to DB
        ExchangeRate::create([
            'from_currency' => $from,
            'to_currency' => $to,
            'rate' => $fetchedRate,
            'rate_date' => Carbon::now(),
        ]);

        return $fetchedRate;
    }
    public function getAllRates(): array
    {

        return Cache::remember('exchange_rates', $this->cacheSeconds, function () {
            return $this->provider->getAllRates();
        });
    }
}
