<?php

namespace App\Services\Sms;

use App\Models\NotificationTemplate;
use App\Services\Sms\SmsSender;

use App\Models\SmsTemplate; // If templates are stored in DB
use App\Models\User;
use Exception;

class NotificationService
{
    protected SmsSender $smsSender;

    public function __construct(SmsSender $smsSender)
    {
        $this->smsSender = $smsSender;
    }

    /**
     * Send SMS to a customer using a template and data.
     */
    public function sendCustomerNotification(User $customer, string $templateKey, array $data): bool
    {
        // Get template (DB or config)
        $template = $this->getTemplate($templateKey);

        if (!$template) {
            throw new Exception("Template '$templateKey' not found.");
        }

        $message = $template['code'] ?? $template;

        // Merge customer-specific data if needed
        $mergedData = array_merge($this->getCustomerData($customer), $data);

        // Send SMS
        return $this->smsSender->send($customer->phone, $message, $mergedData);
    }

    protected function getTemplate(string $title): ?array
    {
        // Example: load from database
        return NotificationTemplate::where('title', $title)->first()?->toArray();

        // Or from a config file:
        // return config("sms_templates.$key");
    }

    protected function getCustomerData(User $customer): array
    {
        return [
            'account_name' => $customer->name,
            'balance' => $customer->balance ?? '0',
            'date' => now()->format('Y-m-d'),

            'currency' => $customer->currency ?? 'TL',
            'pf_name' => optional($customer->portfolio)->name,
            'pr_building_number' => $customer->unit_number,
            'contract_start_date' => optional($customer->contract)->start_date,
            'payment_day' => optional($customer->contract)->payment_day,
            //'payment_link' => route('payment.link', $customer->id), // example
        ];
    }
}
