<?php

declare(strict_types=1);

namespace App\Services\Transfer;

use Illuminate\Support\Str;

use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Http\Controllers\JournalEntryController;
use Abivia\Ledger\Messages\Entry;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\JournalReferenceController;
use Abivia\Ledger\Messages\EntityRef;
use Abivia\Ledger\Messages\Reference;
use Abivia\Ledger\Models\JournalEntry;
use App\Models\Agent;
use App\Models\LedgerBooking;
use App\Models\LedgerExchange;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;
use App\Services\Transfer\Contracts\AccountResolverInterface;
use App\Services\Transfer\Resolvers\StandardAccountResolver;
use App\Services\Transfer\Resolvers\BookingAccountResolver;
use App\Services\Transfer\DTOs\TransferCalculationDTO;

class BookingService
{
  protected ?LedgerBooking $transfer = null;
  protected AccountResolverInterface $accountResolver;

  public function __construct(AccountResolverInterface $accountResolver = null)
  {
    $this->accountResolver = $accountResolver ?? new StandardAccountResolver();
  }

  protected function getAccountResolver(LedgerBooking $transfer): AccountResolverInterface
  {
    // Use booking resolver for booking types, standard for others
    if (strtolower($transfer->type) === 'booking') {
      return new BookingAccountResolver();
    }

    return new StandardAccountResolver();
  }
  public function create(array $data): LedgerBooking
  {

    $data['reference'] = $this->generateRandomTransferNumber();
    $data['secret'] = rand(1000, 9999);
    $data['user_id'] = Auth::id() ?? 1;
    $transfer = LedgerBooking::create($data);

    $transfer->old_transfer = $data['old_transfer'] ?? null;
    return $transfer;
  }


  public function createTransferLedgerEntries(LedgerBooking $transfer): bool
  {

    $this->transfer = $transfer;
    try {

      $transfer->load("OriginalTransfer");
      $originalTransfer = $transfer->OriginalTransfer;
      $exchangeservice = new ExchangeService();
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($transfer);

      Log::info("Creating transfer ledger entries for transfer ID: {$transfer->id}, reference: {$transfer->reference}");

      if ($transfer->type == "Booking" || $transfer->type == "reverse" || $transfer->type == "booking") {

        Log::info("Building booking transfer details for transfer ID: {$transfer->id}, reference: {$transfer->reference}");
        if ($originalTransfer && ($originalTransfer->type == "transfer" || $originalTransfer->type == "approval")) {
          $cash = $this->buildBookingReverseTransferDetails($transfer, $baseReference);
        } else {
          $cash = $this->buildBookingTransferDetails($transfer, $baseReference);
        }
        if ($transfer->currency != $transfer->delivery_currency && $originalTransfer?->type != "transfer") {
          $details = $exchangeservice->BuildExchangeLedgerDetails($transfer->currency, $transfer->delivery_currency, $transfer->amount, $transfer->delivery_amount, $transfer);
          $details = array_merge(
            $cash,
            $details
          );
        } else {
          $details = $cash;
        }











        $groupedByCurrency = collect($details)->groupBy('currency');
        $groupedByCurrency = $groupedByCurrency->toArray();


        foreach ($groupedByCurrency as $currency => $details) {
          $entry_array = Entry::fromArray([
            'currency' => $currency, // 'TRY'
            'entryType' => 'Transfer',
            'clearing' => true,
            'description' => "Transfer {$transfer->reference}",
            'transDate' => $timestamp,
            'clearing' => true,
            'language' => App::getLocale(),
            'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $transfer->id]),
            'details' =>  $details,
          ]);

          if ($transfer->status == "reverse") {
            $entry_array->extra = json_encode([
              'type' => 'reversal',
              'reverses_transfer_id' => $transfer->extra['reverses_transfer_id'] ?? null
            ]);
          }
          $result = $controller->add($entry_array);
          $result->transfer_id = $transfer->id;
          $result->save();
        }
      }

      if ($transfer->status != "reverse") {
        $status = "completed";
        $transfer->status = $status;
      }
      $transfer->save();



      return true;
    } catch (Breaker $e) {
      //  return  false;
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      throw $e;
      return  false;
      // throw $e;
    }
  }



  public function createBaseReference(LedgerBooking $transfer): Reference
  {
    $reference = new Reference();
    $reference->code = $transfer->reference;
    $reference->domain = new EntityRef();
    $reference->domain->code = 'DEFAULT';
    $reference->extra = json_encode([
      'transfer_id' => $transfer->id,
      'type' => 'agent_transfer'
    ]);

    return $reference;
  }





  public function buildBookingTransferDetails(LedgerBooking $transfer, Reference $baseReference): array
  {
    $accounts = $this->getAccountResolver($transfer)->resolveAccounts($transfer);

    $rawDetails = [];

    $cashAmount = $transfer->amount + $transfer->send_fee;
    $company_fee = $transfer->send_fee;

    if ($cashAmount > 0) {
      $rawDetails[] = [
        'code' => $accounts['sender']['cash'],
        'currency' => $transfer->currency,
        'debit' => $cashAmount,
        'extra' => json_encode(['detail_type' => 'INITIAL_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'INITIAL_TRANSFER')],
      ];
    }

    if ($transfer->delivery_amount != 0) {
      $rawDetails[] = [
        'code' => $accounts['receiver']['cash'],
        'currency' => $transfer->delivery_currency,
        'credit' => $transfer->delivery_amount + $transfer->receiver_fee,
        'extra' => json_encode(['detail_type' => 'DELIVERY_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'DELIVERY_TRANSFER')],
      ];
    }
    if ($transfer->currency != $transfer->delivery_currency) {
      if ($company_fee != 0) {
        $rawDetails[] = [
          'code' => $accounts['fees']['commission'],
          'currency' => $transfer->currency,
          'credit' => $company_fee,
          'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
          'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')],
        ];

        if ($transfer->receiver_fee != 0) {
          $rawDetails[] = [
            'code' => $accounts['fees']['commission'],
            'currency' => $transfer->delivery_currency,
            'debit' => $transfer->receiver_fee,
            'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
            'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')],
          ];
        }
      }
    } else {
      if ($company_fee - $transfer->receiver_fee != 0) {
        $rawDetails[] = [
          'code' => $accounts['fees']['commission'],
          'currency' => $transfer->currency,
          'credit' => $company_fee - $transfer->receiver_fee,
          'extra' => json_encode(['detail_type' => 'EARNED_COMMISSION', 'transfer_id' => $transfer->id]),
          'references' => [$this->createDetailReference($baseReference, 'EARNED_COMMISSION')],
        ];
      }
    }


    // 🔁 Merge by account code and currency


    return array_values($rawDetails);
  }
  public function buildBookingReverseTransferDetails(LedgerBooking $transfer, Reference $baseReference): array
  {
    $accounts = $this->getAccountResolver($transfer)->resolveAccounts($transfer);

    $details = [];
    if ($transfer->delivery_amount > 0) {
      $details[] = [
        'code' => $accounts['sender']['not_delivered'], //1300
        'currency' => $transfer->delivery_currency,
        'debit' => $transfer->delivery_amount,
        'extra' => json_encode(['detail_type' => 'REVERSE_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'REVERSE_TRANSFER')],
      ];
    }

    if ($transfer->send_fee > 0) {
      $details[] =   [
        'code' => $accounts['sender']['unearned_revenue'],
        'currency' => $transfer->delivery_currency,
        'debit' => $transfer->send_fee,
        'extra' => json_encode(['detail_type' => 'REVERSE_TRANSFER', 'transfer_id' => $transfer->id]),
        'references' => [$this->createDetailReference($baseReference, 'REVERSE_TRANSFER')],
      ];
    }
    $details[] = [

      'code' => $accounts['receiver']['cash'],
      'currency' => $transfer->delivery_currency,
      'credit' => $transfer->send_fee + $transfer->delivery_amount,
      'extra' => json_encode(['detail_type' => 'REVERSE_TRANSFER', 'transfer_id' => $transfer->id]),
      'references' => [$this->createDetailReference($baseReference, 'REVERSE_TRANSFER')],

    ];

    return $details;
  }



  public  function UpdateLedger(LedgerBooking $transfer)
  {
    try {
      $scheme = request()->isSecure() ? 'https' : 'http';
      $host = request()->getHost();
      $entry = $transfer->journalEntries()->first();
      $entry_id = $entry->journalEntryId;

      $baseReference = $this->createBaseReference($transfer);
      $details = $this->buildBookingTransferDetails($transfer,  $baseReference);
      $extra = json_decode($entry->extra, true) ?? [];
      $extra['is_updated'] = true;
      $response = Http::post("{$scheme}://{$host}/api/ledger/entry/get", [
        'id' => $entry_id,
      ]);
      $ledgerEntry = $response->json();

      $ledgerEntry = $ledgerEntry['entry'];


      $revision = $ledgerEntry['revision'];

      $response = Http::post("{$scheme}://{$host}/api/ledger/entry/update", [
        'id' => $entry_id,
        'revision' => $revision,
        'currency' => $transfer->currency, // or dynamic
        'date' => $transfer->updated_at->toDateString(), // or use now
        'description' => 'Updated Booking #' . $transfer->id,
        'extra' => $extra,

        'details' =>  $details
      ]);

      return (['success' => true, 'message' => __("alerts.saved_successfully")]);
    } catch (\Exception $e) {
      Log::info($e);
      return  ['success' => false, 'message' => __("alerts.something_went_wrong")];
    }
  }
  public  function DeleteLedger(LedgerBooking $transfer)
  {
    try {
      $scheme = request()->isSecure() ? 'https' : 'http';
      $host = request()->getHost();
      $entries = $transfer->journalEntries;
      foreach ($entries as $entry) {
        Log::info($entry);
        $entry_id = $entry->journalEntryId;

        $response = Http::post("{$scheme}://{$host}/api/ledger/entry/get", [
          'id' => $entry_id,
        ]);
        $ledgerEntry = $response->json();

        $ledgerEntry = $ledgerEntry['entry'];


        $revision = $ledgerEntry['revision'];

        $response = Http::post("{$scheme}://{$host}/api/ledger/entry/delete", [
          'id' => $entry_id,
          'revision' => $revision,

        ]);
      }
      return (['success' => true, 'message' => __("alerts.saved_successfully")]);
    } catch (\Exception $e) {
      Log::info($e);
      return  ['success' => false, 'message' => __("alerts.something_went_wrong")];
    }
  }



  public function debugBalance(string $title, array $details)
  {
    $sum = 0;
    $text = "$title Details:\n";

    foreach ($details as $d) {
      $amount = $d['debit'] ?? ($d['credit'] ?? 0);
      $type = isset($d['debit']) ? 'Debit' : 'Credit';
      $sum += isset($d['debit']) ? $amount : -$amount;

      // Fetch the account name
      $account = LedgerAccount::with("names")->where('code', $d['code'])->first();

      $accountName = $account && $account->names->isNotEmpty() ? $account->names->first()->name : 'Unknown Account';

      $text .= "- {$d['code']} ({$accountName}): {$type} {$amount}\n";
    }

    $text .= "Total Balance: $sum\n";
    $text .= "------------------\n";

    return $text;
  }


  public function createDetailReference(Reference $baseReference, string $suffix): Reference
  {
    $reference = new Reference();
    $reference->code = $baseReference->code . '-' . $suffix;
    $reference->domain = clone $baseReference->domain;
    $reference->extra = json_encode(array_merge(
      json_decode($baseReference->extra, true) ?? [],
      ['detail_type' => $suffix]
    ));

    //     $controller = new JournalReferenceController();
    //    $reference = $controller->add($reference);


    return $reference;
  }










  public function generateRandomTransferNumber(): string
  {
    $branchCode = str_pad((string)mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad((string)mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "REC-{$branchCode}-{$transactionId}";
  }
}
