<?php

namespace App\Services;

use Abivia\Ledger\Messages\Detail;
use Abivia\Ledger\Messages\Entry;
use App\Models\Agent;
use App\Models\LedgerTransfer;
use App\Models\Transfer;
use Illuminate\Support\Facades\DB;

class TransferService
{
    public function sendMoney(
        Agent $sender,
        Agent $receiver,
        float $amount,
        float $fee = 0,
        array $customerDetails = []
    ): Transfer {
        return DB::transaction(function () use ($sender, $receiver, $amount, $fee, $customerDetails) {
            // Create the transfer record
            $transfer = Transfer::create([
                'sender_id' => $sender->id,
                'receiver_id' => $receiver->id,
                'amount' => $amount,
                'fee' => $fee,
                'status' => 'pending',
                'reference' => $this->generateReference(),
                'customer_details' => $customerDetails,
            ]);

            // Record in ledger
            $this->recordLedgerEntries($transfer);

            // Update status
            $transfer->update(['status' => 'completed']);

            return $transfer;
        });
    }

    protected function recordLedgerEntries(LedgerTransfer $transfer)
    {
        $entry = new Entry();
        $entry->currency = $transfer->currency;
        $entry->description = "Transfer {$transfer->reference}";

        // Debit receiver (funds owed to them)
        $detail = new Detail();
        $detail->amount = $transfer->amount;
        $detail->ledgerUuid = $this->getAccountUuid('2100'); // Obligations
        $detail->journalReferenceUuid = $transfer->receiver->ledger_uuid;
        $entry->addDetail($detail);

        // Credit sender (funds from them)
        $detail = new Detail();
        $detail->amount = -$transfer->amount;
        $detail->ledgerUuid = $this->getAccountUuid('1100'); // Agent Funds
        $detail->journalReferenceUuid = $transfer->sender->ledger_uuid;
        $entry->addDetail($detail);

        // Record fee if applicable
        if ($transfer->fee > 0) {
            $detail = new Detail();
            $detail->amount = $transfer->fee;
            $detail->ledgerUuid = $this->getAccountUuid('4100'); // Fee Income
            $entry->addDetail($detail);

            $detail = new Detail();
            $detail->amount = -$transfer->fee;
            $detail->ledgerUuid = $this->getAccountUuid('1100'); // Agent Funds
            $detail->journalReferenceUuid = $transfer->sender->ledger_uuid;
            $entry->addDetail($detail);
        }

        $controller = new \Abivia\Ledger\Http\Controllers\JournalEntryController();
        $response = $controller->run($entry);

        $transfer->update([
            'ledger_journal_uuid' => $response->journalEntry->journalReferenceUuid
        ]);
    }

    protected function getAccountUuid(string $code): string
    {
        return \Abivia\Ledger\Models\LedgerAccount::where('code', $code)
            ->first()
            ->ledgerUuid;
    }

    protected function generateReference(): string
    {
        return 'TX' . now()->format('YmdHis') . rand(100, 999);
    }
}
