<?php

namespace App\Services\WhatsApp;

use App\Services\Contracts\NotificationStrategy;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class WhatsappSender implements NotificationStrategy
{
    protected $config;

    public function __construct()
    {
        $this->config = [
            'url' => env('WHATSAPP_API_URL'),
            'token' => env('WHATSAPP_API_TOKEN'),
            'sender_id' => env('WHATSAPP_SENDER_ID'),
        ];

        if (!$this->config['url'] || !$this->config['token']) {
            throw new Exception("WhatsApp configuration is missing.");
        }
    }

    public function send(string $to, string $message, array $meta = []): bool
    {
        $payload = [
            'to' => $to,
            'message' => $message,
            'sender_id' => $this->config['sender_id'] ?? null,
        ];

        $response = Http::withToken($this->config['token'])
            ->post($this->config['url'], $payload);

        if ($response->successful()) {
            Log::info('WhatsApp message sent', ['to' => $to, 'response' => $response->body()]);
            return true;
        }

        Log::error('WhatsApp sending failed', [
            'to' => $to,
            'payload' => $payload,
            'response' => $response->body(),
        ]);

        throw new Exception("WhatsApp sending failed. Response: " . $response->body());
    }
}
