<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('ledger_transfers', function (Blueprint $table) {
            // Safely drop foreign keys if they exist
            $this->dropForeignIfExists('ledger_transfers', 'ledger_transfers_sender_id_foreign');
            $this->dropForeignIfExists('ledger_transfers', 'ledger_transfers_receiver_id_foreign');
        });

        Schema::table('ledger_transfers', function (Blueprint $table) {
            // Alter columns to nullable
            $table->foreignId('sender_id')->nullable()->change();
            $table->foreignId('receiver_id')->nullable()->change();

            // Re-add foreign keys with nullOnDelete
            $table->foreign('sender_id')->references('id')->on('agents')->nullOnDelete();
            $table->foreign('receiver_id')->references('id')->on('agents')->nullOnDelete();

            // Add new columns
            $table->string('sender_account_code')->nullable();
            $table->string('receiver_account_code')->nullable();
        });
    }

    public function down(): void
    {
        Schema::table('ledger_transfers', function (Blueprint $table) {
            $this->dropForeignIfExists('ledger_transfers', 'ledger_transfers_sender_id_foreign');
            $this->dropForeignIfExists('ledger_transfers', 'ledger_transfers_receiver_id_foreign');
        });

        Schema::table('ledger_transfers', function (Blueprint $table) {
            // Revert sender/receiver back to NOT NULL
            $table->foreignId('sender_id')->nullable(false)->change();
            $table->foreignId('receiver_id')->nullable(false)->change();

            // Re-add restrictive FKs
            $table->foreign('sender_id')->references('id')->on('agents')->restrictOnDelete();
            $table->foreign('receiver_id')->references('id')->on('agents')->restrictOnDelete();

            // Drop added columns
            $table->dropColumn('sender_account_code');
            $table->dropColumn('receiver_account_code');
        });
    }

    /**
     * Helper: drop a foreign key if it exists
     */
    private function dropForeignIfExists(string $table, string $foreignKey): void
    {
        $exists = DB::selectOne("
            SELECT COUNT(*) as cnt
            FROM information_schema.REFERENTIAL_CONSTRAINTS
            WHERE CONSTRAINT_SCHEMA = DATABASE()
              AND CONSTRAINT_NAME = ?
              AND TABLE_NAME = ?
        ", [$foreignKey, $table]);

        if ($exists && $exists->cnt > 0) {
            DB::statement("ALTER TABLE `$table` DROP FOREIGN KEY `$foreignKey`");
        }
    }
};
